<?php

namespace MediaWiki\CheckUser\Logging;

use MediaWiki\Extension\AbuseFilter\ProtectedVarsAccessLogger;
use MediaWiki\Linker\Linker;
use MediaWiki\Logging\LogEntry;
use MediaWiki\Logging\LogFormatter;
use MediaWiki\MediaWikiServices;
use MediaWiki\Message\Message;
use MediaWiki\User\UserFactory;

class TemporaryAccountLogFormatter extends LogFormatter {

	private UserFactory $userFactory;

	public function __construct(
		LogEntry $entry,
		UserFactory $userFactory
	) {
		parent::__construct( $entry );
		$this->userFactory = $userFactory;
	}

	/**
	 * @inheritDoc
	 */
	protected function getMessageParameters() {
		$params = parent::getMessageParameters();

		// Update the logline depending on if the user had their access enabled or disabled
		if ( $this->entry->getSubtype() === TemporaryAccountLogger::ACTION_CHANGE_ACCESS ) {
			// Message keys used:
			// - 'checkuser-temporary-account-change-access-level-enable'
			// - 'checkuser-temporary-account-change-access-level-disable'
			$params[3] = $this->msg( 'checkuser-temporary-account-change-access-level-' . $params[3], $params[1] );
		} elseif ( $this->entry->getSubtype() === TemporaryAccountLogger::ACTION_CHANGE_AUTO_REVEAL ) {
			if ( $params[3] === TemporaryAccountLogger::ACTION_AUTO_REVEAL_ENABLED ) {
				$params[4] = Message::dateTimeParam( $params[4] );
			}
		} elseif ( $this->entry->getSubtype() === TemporaryAccountLogger::ACTION_VIEW_IPS ) {
			// Replace temporary user page link with contributions page link.
			// Don't use LogFormatter::makeUserLink, because that adds tools links.
			$tempUserName = $this->entry->getTarget()->getText();
			$params[2] = Message::rawParam(
				Linker::userLink( 0, $this->userFactory->newUnsavedTempUser( $tempUserName ) )
			);
		} elseif (
			$this->entry->getSubtype() === TemporaryAccountLogger::ACTION_VIEW_TEMPORARY_ACCOUNTS_ON_IP ||
			$this->entry->getSubtype() === TemporaryAccountLogger::ACTION_VIEW_TEMPORARY_ACCOUNTS_ON_IP_GLOBAL
		) {
			// Replace IP address with contributions page link generated by Linker::userLink.
			$ip = $this->entry->getTarget()->getText();
			$params[2] = Message::rawParam( Linker::userLink( 0, $ip ) );
		}

		if ( MediaWikiServices::getInstance()->getExtensionRegistry()->isLoaded( 'Abuse Filter' ) ) {
			// Modify external log af-view-protected-var-value
			if (
				$this->entry->getSubtype() === 'af-' . ProtectedVarsAccessLogger::ACTION_VIEW_PROTECTED_VARIABLE_VALUE
			) {
				// Replace temporary user page link with contributions page link.
				// Don't use LogFormatter::makeUserLink, because that adds tools links.
				$tempUserName = $this->entry->getTarget()->getText();
				$params[2] = Message::rawParam(
					Linker::userLink( 0, $this->userFactory->newUnsavedTempUser( $tempUserName ) )
				);
			}
		}

		return $params;
	}

	/**
	 * @inheritDoc
	 */
	protected function getMessageKey() {
		if ( $this->entry->getSubtype() === TemporaryAccountLogger::ACTION_CHANGE_AUTO_REVEAL ) {
			$params = $this->getMessageParameters();
			if ( $params[3] === TemporaryAccountLogger::ACTION_AUTO_REVEAL_ENABLED ) {
				return 'logentry-checkuser-temporary-account-enable-auto-reveal';
			} elseif ( $params[3] === TemporaryAccountLogger::ACTION_AUTO_REVEAL_DISABLED ) {
				return 'logentry-checkuser-temporary-account-disable-auto-reveal';
			}
		} else {
			return parent::getMessageKey();
		}
	}
}
