// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/UnstableParticles.hh"

namespace Rivet {


  /// @brief eta, eta' and phi in D0, D+, Ds decays
  class CLEOC_2006_I728043 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(CLEOC_2006_I728043);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {
      // projection
      declare(UnstableParticles(),"UFS");
      // histograms
      for (double eVal : allowedEnergies()) {
        const string en = toString(round(eVal/MeV));
        if (isCompatibleWithSqrtS(eVal, 1e-3))  _sqs = en;
        size_t imin = 0, imax = 2;
        if (en == "4170"s) {
          imin = 2, imax = 3;
        }
        for (size_t ix=imin; ix<imax; ++ix) {
          const string id = toString(ix);
          book(_n[id], "TMP/n_D_"+id);
          book(_c["eta"+id], 1,1,ix+1);
          book(_c["etaPrime"+id], 2,1,ix+1);
          book(_c["phi"+id], 3,1,ix+1);
          book(_h["eta"+id], 4,1,ix+1);
          book(_h["phi"+id], 5,1,ix+1);
        }
      }
      raiseBeamErrorIf(_sqs.empty());
    }

    void fillHistos(const Particle& Dmeson, const LorentzTransform& boost) {
      Particles ssbar;
      unsigned int iMeson=0;
      if (Dmeson.abspid()==421)      iMeson = 1;
      else if (Dmeson.abspid()==431) iMeson = 2;
      string id = toString(iMeson);
      _n[id]->fill();
      findDecayProducts(Dmeson,ssbar);
      for (const Particle& dec : ssbar) {
        FourMomentum p = boost.transform(dec.mom());
        double mom=p.p3().mod();
        if (dec.pid()==221) {
          _c["eta"+id]->fill();
          _h["eta"+id]->fill(mom);
        }
        else if (dec.pid()==331) {
          _c["etaPrime"+id]->fill();
        }
        else {
          _c["phi"+id]->fill();
          _h["phi"+id] ->fill(mom);
        }
      }
    }

    void findDecayProducts(const Particle& mother, Particles& ssbar) const {
      for (const Particle& p : mother.children()) {
        int id = p.pid();
      	if (id==221 || id==331 || id==333) {
          ssbar.push_back(p);
          findDecayProducts(p,ssbar);
        }
        else if ( !p.children().empty() ) {
          findDecayProducts(p,ssbar);
        }
      }
    }

    /// Perform the per-event analysis
    void analyze(const Event& event) {
      // find psi(3770)
      const UnstableParticles& ufs = apply<UnstableParticles>(event, "UFS");
      Particles psi = ufs.particles(Cuts::pid==30443);
      // D_s
      if (psi.empty()) {
        LorentzTransform boost;
        for (const Particle& Dmeson : apply<UnstableParticles>("UFS",event).particles(Cuts::abspid==431)) {
          fillHistos(Dmeson,boost);
        }
      }
      // D0 D+
      else {
        for (const Particle& p : psi) {
          // boost to rest frame
          LorentzTransform boost;
          if (p.p3().mod() > 1*MeV) {
            boost = LorentzTransform::mkFrameTransformFromBeta(p.momentum().betaVec());
          }
          // loop over D0 and D+ children
          for (const Particle& Dmeson : p.children()) {
            if (Dmeson.abspid() != 411 && Dmeson.abspid() != 421) continue;
            fillHistos(Dmeson,boost);
          }
        }
      }
    }


    /// Normalise histograms etc., after the run
    void finalize() {
      for (const auto& item : _n) {
        if (item.second->effNumEntries() <= 0.)  continue;
        const double sf = 100./ item.second->sumW();
        scale(_c["eta"+item.first], sf);
        scale(_c["etaPrime"+item.first], sf);
        scale(_c["phi"+item.first], sf);
        scale(_h["eta"+item.first], sf);
        scale(_h["phi"+item.first], sf);
      }
    }

    /// @}


    /// @name Histograms
    /// @{
    map<string,CounterPtr> _c, _n;
    map<string,Histo1DPtr> _h;

    string _sqs = "";
    /// @}


  };


  RIVET_DECLARE_PLUGIN(CLEOC_2006_I728043);

}
