// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/UnstableParticles.hh"

namespace Rivet {


  /// @brief Lambda_b asymmetry
  class LHCB_2021_I1888216 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(LHCB_2021_I1888216);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {
      // projection
      declare(UnstableParticles(), "UFS");
      //histograms
      for (double eVal : allowedEnergies()) {
        const int en(eVal+0.5);
        if (isCompatibleWithSqrtS(eVal))  _sqs = en;
        size_t ih(en == 8000);
        book(_e[en], 1, 1, ih+1);
        book(_e[en+1], 2, 1, ih+1);
        for (size_t ix=0; ix<2; ++ix) {
          book(_h_y[en+ix],  "TMP/h_y_ "+toString(en+ix+1), refData(1,1,1));
          book(_h_pT[en+ix], "TMP/h_pT_"+toString(en+ix+1), refData(2,1,1));
        }
      }
      raiseBeamErrorIf(_sqs==0);
    }


    /// Perform the per-event analysis
    void analyze(const Event& event) {
      const UnstableParticles& ufs = apply<UnstableParticles>(event, "UFS");
      for (const Particle& p : ufs.particles(Cuts::abspid==5122)) {
        const double pT = p.perp()/GeV;
        const double y  = p.absrap();
        if (y<2.15 ||  y>4.1) continue;
        if (pT<2   || pT>27 ) continue;
        bool anti = p.pid()<0;
        _h_pT[_sqs+anti]->fill(pT);
        _h_y [_sqs+anti]->fill(y);
      }
    }


    /// Normalise histograms etc., after the run
    void finalize() {
      // calculate asymmetry in %
      for (double eVal : allowedEnergies()) {
        const int en(eVal+0.5);
        asymm(_h_y[en+0], _h_y[en+1], _e[en]);
        asymm(_h_pT[en+0], _h_pT[en+1], _e[en+1]);
      }
      scale(_e, 100.);
    }

    /// @}


    /// @name Histograms
    /// @{
    map<int,Histo1DPtr> _h_pT, _h_y;
    map<int,Estimate1DPtr> _e;
    int _sqs = 0;
    /// @}


  };


  RIVET_DECLARE_PLUGIN(LHCB_2021_I1888216);

}
