// -*- C++ -*-
#ifndef RIVET_METFinder_HH
#define RIVET_METFinder_HH

#include "Rivet/Projection.hh"

namespace Rivet {


  /// Interface for projections that find missing transverse energy/momentum
  class METFinder : public Projection {
  public:

    /// Import to avoid warnings about overload-hiding
    using Projection::operator=;

    /// @name Transverse momentum functions
    ///
    /// @note This may be what you want, even if the paper calls it "missing Et"!
    /// @{

    /// The vector-summed visible transverse momentum in the event, as a 3-vector with z=0
    ///
    /// @note Reverse this vector with operator- to get the missing pT vector.
    /// @todo Currently equivalent to vectorEt
    virtual const Vector3& vectorSumPt() const = 0;
    /// Alias for vectorSumPt
    virtual const Vector3& vectorPt() const { return vectorSumPt(); }

    /// Convenience vector MPT function
    const Vector3 vectorMissingPt() const { return -vectorSumPt(); }
    // Alias
    const Vector3 vectorPtMiss() const { return vectorMissingPt(); }
    // Alias
    const Vector3 vectorMPT() const { return vectorMissingPt(); }

    /// The vector-summed missing transverse momentum in the event.
    double missingPt() const { return vectorPt().mod(); }
    /// Alias for missingPt
    double pTmiss() const { return missingPt(); }
    /// Alias for missingPt
    double scalarPtMiss() const { return missingPt(); }
    /// Alias for missingPt
    double mpt() const { return missingPt(); }

    /// The scalar-summed visible transverse momentum in the event.
    virtual double scalarSumPt() const = 0;
    /// Alias for scalarSumPt
    double scalarPt() const { return scalarSumPt(); }
    /// Alias for scalarSumPt
    double spt() const { return scalarSumPt(); }

    /// @}


    /// @name Transverse energy functions
    ///
    /// @warning Despite the common names "MET" and "SET", what's often meant is the pT functions above!
    /// @{

    /// The vector-summed visible transverse energy in the event, as a 3-vector with z=0
    ///
    /// @note Reverse this vector with operator- to get the missing ET vector.
    virtual const Vector3& vectorSumEt() const = 0;
    /// Alias for vectorSumEt
    virtual const Vector3& vectorEt() const { return vectorSumEt(); }

    /// Convenience vector MET function
    const Vector3 vectorMissingEt() const { return -vectorSumEt(); }
    // Alias
    const Vector3 vectorEtMiss() const { return vectorMissingEt(); }
    // Alias
    const Vector3 vectorMET() const { return vectorMissingEt(); }

    /// The vector-summed missing transverse energy in the event.
    double missingEt() const { return vectorMissingEt().mod(); }
    /// Alias for missingPt
    double eTmiss() const { return missingEt(); }
    /// Alias for missingPt
    double scalarEtMiss() const { return missingEt(); }
    /// Alias for missingEt
    double met() const { return missingEt(); }

    /// The scalar-summed visible transverse energy in the event.
    virtual double scalarSumEt() const = 0;
    /// Alias for scalarSumEt
    double scalarEt() const { return scalarSumEt(); }
    /// Alias for scalarSumEt
    double set() const { return scalarSumEt(); }

    /// @}


    /// Reset the projection. Smearing functions will be unchanged.
    virtual void reset() {  }

  };


}

#endif
