// Licensed to the Software Freedom Conservancy (SFC) under one
// or more contributor license agreements.  See the NOTICE file
// distributed with this work for additional information
// regarding copyright ownership.  The SFC licenses this file
// to you under the Apache License, Version 2.0 (the
// "License"); you may not use this file except in compliance
// with the License.  You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.

package org.openqa.selenium.docker.client;

import java.time.Duration;
import java.util.Set;
import java.util.logging.Logger;
import org.openqa.selenium.docker.Container;
import org.openqa.selenium.docker.ContainerConfig;
import org.openqa.selenium.docker.ContainerId;
import org.openqa.selenium.docker.ContainerInfo;
import org.openqa.selenium.docker.ContainerLogs;
import org.openqa.selenium.docker.DockerException;
import org.openqa.selenium.docker.DockerProtocol;
import org.openqa.selenium.docker.Image;
import org.openqa.selenium.docker.internal.Reference;
import org.openqa.selenium.internal.Require;
import org.openqa.selenium.remote.http.HttpHandler;

public class DockerClient implements DockerProtocol {

  private static final Logger LOG = Logger.getLogger(DockerClient.class.getName());
  private final String apiVersion;
  private final ApiVersionAdapter adapter;
  private final ListImages listImages;
  private final PullImage pullImage;
  private final CreateContainer createContainer;
  private final StartContainer startContainer;
  private final StopContainer stopContainer;
  private final IsContainerPresent isContainerPresent;
  private final InspectContainer inspectContainer;
  private final GetContainerLogs containerLogs;

  public DockerClient(HttpHandler client, String apiVersion) {
    Require.nonNull("HTTP client", client);
    this.apiVersion = Require.nonNull("API version", apiVersion);
    this.adapter = AdapterFactory.createAdapter(apiVersion);

    LOG.info(
        "Initialized DockerClient with API version "
            + apiVersion
            + " (adapter: "
            + adapter.getClass().getSimpleName()
            + ")");

    listImages = new ListImages(client, apiVersion, adapter);
    pullImage = new PullImage(client, apiVersion);

    createContainer = new CreateContainer(this, client, apiVersion, adapter);
    startContainer = new StartContainer(client, apiVersion);
    stopContainer = new StopContainer(client, apiVersion);
    isContainerPresent = new IsContainerPresent(client, apiVersion);
    inspectContainer = new InspectContainer(client, apiVersion, adapter);
    containerLogs = new GetContainerLogs(client, apiVersion);
  }

  /**
   * Returns the adapter for this API version.
   *
   * @return The API version adapter
   */
  public ApiVersionAdapter getAdapter() {
    return adapter;
  }

  @Override
  public String version() {
    return apiVersion;
  }

  @Override
  public Image getImage(String imageName) throws DockerException {
    Require.nonNull("Image name", imageName);

    Reference ref = Reference.parse(imageName);

    LOG.info("Listing local images: " + ref);
    Set<Image> allImages = listImages.apply(ref);
    if (!allImages.isEmpty()) {
      return allImages.iterator().next();
    }

    LOG.info("Pulling " + ref);
    pullImage.apply(ref);

    LOG.info("Pull completed. Listing local images again: " + ref);
    allImages = listImages.apply(ref);
    if (!allImages.isEmpty()) {
      return allImages.iterator().next();
    }

    throw new DockerException(
        "Pull appears to have succeeded, but image not present locally: " + imageName);
  }

  @Override
  public Container create(ContainerConfig config) {
    Require.nonNull("Container config", config);

    LOG.fine("Creating container: " + config);

    return createContainer.apply(config);
  }

  @Override
  public boolean isContainerPresent(ContainerId id) throws DockerException {
    Require.nonNull("Container id", id);

    LOG.info("Checking if container is present: " + id);

    return isContainerPresent.apply(id);
  }

  @Override
  public void startContainer(ContainerId id) throws DockerException {
    Require.nonNull("Container id", id);

    LOG.fine("Starting container: " + id);

    startContainer.apply(id);
  }

  @Override
  public void stopContainer(ContainerId id, Duration timeout) throws DockerException {
    Require.nonNull("Container id", id);
    Require.nonNull("Timeout", timeout);

    LOG.fine("Stopping container: " + id);

    stopContainer.apply(id, timeout);
  }

  @Override
  public ContainerInfo inspectContainer(ContainerId id) throws DockerException {
    Require.nonNull("Container id", id);

    LOG.fine("Inspecting container: " + id);

    return inspectContainer.apply(id);
  }

  @Override
  public ContainerLogs getContainerLogs(ContainerId id) throws DockerException {
    Require.nonNull("Container id", id);

    LOG.info("Getting container logs: " + id);

    return containerLogs.apply(id);
  }
}
