// <copyright file="EdgeOptions.cs" company="Selenium Committers">
// Licensed to the Software Freedom Conservancy (SFC) under one
// or more contributor license agreements.  See the NOTICE file
// distributed with this work for additional information
// regarding copyright ownership.  The SFC licenses this file
// to you under the Apache License, Version 2.0 (the
// "License"); you may not use this file except in compliance
// with the License.  You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.
// </copyright>

using System.Globalization;
using OpenQA.Selenium.Chromium;

namespace OpenQA.Selenium.Edge;

/// <summary>
/// Class to manage options specific to <see cref="EdgeDriver"/>
/// </summary>
/// <example>
/// <code>
/// EdgeOptions options = new EdgeOptions();
/// </code>
/// <para></para>
/// <para>For use with EdgeDriver:</para>
/// <para></para>
/// <code>
/// EdgeDriver driver = new EdgeDriver(options);
/// </code>
/// <para></para>
/// <para>For use with RemoteWebDriver:</para>
/// <para></para>
/// <code>
/// RemoteWebDriver driver = new RemoteWebDriver(new Uri("http://localhost:4444/wd/hub"), options.ToCapabilities());
/// </code>
/// </example>
public class EdgeOptions : ChromiumOptions
{
    private const string DefaultBrowserNameValue = "MicrosoftEdge";
    private const string WebViewBrowserNameValue = "webview2";
    private const string EdgeOptionsCapabilityName = "edgeOptions";

    /// <summary>
    /// Initializes a new instance of the <see cref="EdgeOptions"/> class.
    /// </summary>
    public EdgeOptions() : base()
    {
        this.BrowserName = DefaultBrowserNameValue;
    }

    /// <summary>
    /// Gets the vendor prefix to apply to Chromium-specific capability names.
    /// </summary>
    protected override string VendorPrefix => "ms";

    /// <summary>
    /// Gets the name of the capability used to store Chromium options in
    /// an <see cref="ICapabilities"/> object.
    /// </summary>
    public override string CapabilityName => string.Format(CultureInfo.InvariantCulture, "{0}:{1}", this.VendorPrefix, EdgeOptionsCapabilityName);

    /// <summary>
    /// Gets or sets whether to create a WebView session used for launching an Edge (Chromium) WebView-based app on desktop.
    /// </summary>
    public bool UseWebView
    {
        get => this.BrowserName == WebViewBrowserNameValue;
        set => this.BrowserName = value ? WebViewBrowserNameValue : DefaultBrowserNameValue;
    }

    /// <summary>
    /// Provides a means to add additional capabilities not yet added as type safe options
    /// for the Edge driver.
    /// </summary>
    /// <param name="optionName">The name of the capability to add.</param>
    /// <param name="optionValue">The value of the capability to add.</param>
    /// <exception cref="ArgumentException">
    /// thrown when attempting to add a capability for which there is already a type safe option, or
    /// when <paramref name="optionName"/> is <see langword="null"/> or the empty string.
    /// </exception>
    /// <remarks>Calling <see cref="AddAdditionalEdgeOption(string, object)"/>
    /// where <paramref name="optionName"/> has already been added will overwrite the
    /// existing value with the new value in <paramref name="optionValue"/>.
    /// Calling this method adds capabilities to the Edge-specific options object passed to
    /// WebDriver executable (property name 'ms:edgeOptions').</remarks>
    public void AddAdditionalEdgeOption(string optionName, object optionValue)
    {
        this.AddAdditionalChromiumOption(optionName, optionValue);
    }
}
