/* ScummVM - Graphic Adventure Engine
 *
 * ScummVM is the legal property of its developers, whose names
 * are too numerous to list here. Please refer to the COPYRIGHT
 * file distributed with this source distribution.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include "access/amazon/amazon_resources.h"
#include "access/access.h"

namespace Access {

namespace Amazon {

static const int BTN_RANGES[6][2] = {
	{ 0, 76 }, { 77, 154 }, { 155, 232 }, { 233, 276 }, { 0, 0 }, { 277, 319 }
};

static const int RMOUSE[10][2] = {
	{ 0, 35 }, { 0, 0 }, { 36, 70 }, { 71, 106 }, { 107, 141 },
	{ 142, 177 }, { 178, 212 }, { 213, 248 }, { 249, 283 }, { 284, 318 }
};

AmazonResources::~AmazonResources() {
	delete _font3x5;
	delete _font6x6;
}

void AmazonResources::load(Common::SeekableReadStream &s) {
	Resources::load(s);
	uint count;

	// Load the version specific data
	NO_HELP_MESSAGE = s.readString();
	NO_HINTS_MESSAGE = s.readString();
	RIVER_HIT1 = s.readString();
	RIVER_HIT2 = s.readString();
	BAR_MESSAGE = s.readString();

	for (int idx = 0; idx < 3; ++idx)
		HELPLVLTXT[idx] = s.readString();
	for (int idx = 0; idx < 9; ++idx)
		IQLABELS[idx] = s.readString();

	CANT_GET_THERE = s.readString();

	// Get the offset of the general shared data for the game
	uint entryOffset = findEntry(_vm->getGameID(), 2, 0, (Common::Language)0);
	s.seek(entryOffset);

	// Read in the cursor list
	count = s.readUint16LE();
	CURSORS.resize(count);
	for (uint idx = 0; idx < count; ++idx) {
		uint count2 = s.readUint16LE();
		CURSORS[idx].resize(count2);
		s.read(&CURSORS[idx][0], count2);
	}

	// Load font data
	count = s.readUint16LE();
	Common::Array<int> index;
	Common::Array<byte> data;

	index.resize(count);
	for (uint idx = 0; idx < count; ++idx)
		index[idx] = s.readSint16LE();

	count = s.readUint16LE();
	data.resize(count);
	for (uint idx = 0; idx < count; ++idx)
		data[idx] = s.readByte();
	_font3x5 = new AmazonFont(&index[0], &data[0]);

	count = s.readUint16LE();
	index.resize(count);
	for (uint idx = 0; idx < count; ++idx)
		index[idx] = s.readSint16LE();

	count = s.readUint16LE();
	data.resize(count);
	for (uint idx = 0; idx < count; ++idx)
		data[idx] = s.readByte();
	_font6x6 = new AmazonFont(&index[0], &data[0]);
}

const byte *AmazonResources::getCursor(int num) const {
	return CURSORS[num].data();
}

int AmazonResources::getRMouse(int i, int j) const {
	return RMOUSE[i][j];
}

int AmazonResources::inButtonXRange(int x) const {
	for (int i = 0; i < ARRAYSIZE(BTN_RANGES); i++) {
		if ((x >= BTN_RANGES[i][0]) && (x < BTN_RANGES[i][1]))
			return i;
	}
	return -1;
}

/*------------------------------------------------------------------------*/

const int SIDEOFFR[] = { 5, 5, 5, 5, 5, 5, 5, 5, 0 };
const int SIDEOFFL[] = { 5, 5, 5, 5, 5, 5, 5, 5, 0 };
const int SIDEOFFU[] = { 2, 2, 2, 2, 2, 2, 2, 2, 0 };
const int SIDEOFFD[] = { 2, 2, 2, 2, 2, 2, 2, 2, 0 };
const int DIAGOFFURX[] = { 4, 5, 2, 2, 3, 4, 2, 2, 0 };
const int DIAGOFFURY[] = { 2, 3, 2, 2, 2, 3, 1, 1, 0 };
const int DIAGOFFDRX[] = { 4, 5, 4, 3, 5, 4, 5, 1, 0 };
const int DIAGOFFDRY[] = { 3, 2, 1, 2, 2, 1, 2, 1, 0 };
const int DIAGOFFULX[] = { 4, 5, 4, 3, 3, 2, 2, 2, 0 };
const int DIAGOFFULY[] = { 3, 3, 1, 2, 2, 1, 1, 1, 0 };
const int DIAGOFFDLX[] = { 4, 5, 3, 3, 5, 4, 6, 1, 0 };
const int DIAGOFFDLY[] = { 2, 2, 1, 2, 3, 1, 2, 1, 0 };

const int OVEROFFR[] = { 2, 2, 1, 2, 2, 1, 0, 0, 0 };
const int OVEROFFL[] = { 2, 2, 1, 2, 2, 1, 0, 0, 0 };
const int OVEROFFU[] = { 1, 1, 1, 1, 1, 1, 0, 0, 0 };
const int OVEROFFD[] = { 1, 1, 1, 1, 1, 1, 0, 0, 0 };
const int OVEROFFURX[] = { 3, 1, 1, 2, 2, 1, 0, 0, 0 };
const int OVEROFFURY[] = { 1, 0, 0, 1, 1, 0, 0, 0, 0 };
const int OVEROFFDRX[] = { 1, 2, 1, 1, 2, 1, 0, 0, 0 };
const int OVEROFFDRY[] = { 0, 1, 0, 0, 1, 1, 0, 0, 0 };
const int OVEROFFULX[] = { 2, 1, 1, 1, 2, 1, 0, 0, 0 };
const int OVEROFFULY[] = { 1, 0, 0, 2, 1, 0, 0, 0, 0 };
const int OVEROFFDLX[] = { 1, 2, 1, 1, 2, 1, 0, 0, 0 };
const int OVEROFFDLY[] = { 0, 1, 0, 0, 1, 1, 0, 0, 0 };

const int DEATH_CELLS[13][3] = {
	{ 0, 94, 2 },
	{ 0, 94, 3 },
	{ 0, 94, 4 },
	{ 0, 94, 5 },
	{ 0, 94, 6 },
	{ 0, 94, 7 },
	{ 0, 94, 8 },
	{ 0, 94, 9 },
	{ 0, 94, 10 },
	{ 0, 94, 11 },
	{ 0, 94, 12 },
	{ 0, 94, 13 },
	{ 0, 94, 14 }
};

const int CHAPTER_CELLS[17][3] = {
	{ 1, 96, 18 },
	{ 2, 96, 19 },
	{ 3, 96, 20 },
	{ 4, 96, 21 },
	{ 5, 96, 22 },
	{ 6, 96, 23 },
	{ 7, 96, 24 },
	{ 8, 96, 25 },
	{ 9, 96, 26 },
	{ 10, 96, 27 },
	{ 11, 96, 28 },
	{ 12, 96, 29 },
	{ 13, 96, 30 },
	{ 14, 96, 31 }
};

const int CHAPTER_TABLE[14][5] = {
	{ 18, 136, 27, 76, 49 },
	{ 16, 134, 27, 53, 74 },
	{ 16, 136, 27, 52, 56 },
	{ 16, 135, 26, 46, 75 },
	{ 16, 135, 27, 54, 66 },
	{ 16, 137, 27, 67, 79 },
	{ 14, 136, 27, 82, 52 },
	{ 15, 136, 26, 65, 73 },
	{ 15, 137, 26, 48, 75 },
	{ 17, 135, 27, 52, 66 },
	{ 15, 135, 27, 62, 65 },
	{ 16, 135, 28, 45, 66 },
	{ 16, 135, 28, 36, 67 },
	{ 15, 135, 27, 34, 63 }
};

const int CHAPTER_JUMP[14] = {
	0, 12, 10, 15, 19, 25, 31, 36, 45, 46, 29, 55, 61, 0
};

const int ANTWALK[24] = {
	 0,  3,  0,
	 1,  5,  0,
	 2,  4,  0,
	 3,  2,  0,
	 4,  4,  0,
	 5,  3,  0,
	 6,  4,  0,
	-1, -1, -1
};

const int ANTEAT[33] = {
	 7,  0,  -1,
	 8,  0,  -5,
	 9,  0, -11,
	10,  0,   7,
	11,  0,  -3,
	12,  0,   3,
	13,  0,  -1,
	 9,  0,  -6,
	 8,  0,  11,
	 7,  0,   6,
	-1, -1,  -1
};

const int ANTDIE[21] = {
	14,  4,  8,
	15,  7,  6,
	16,  6,  7,
	17,  8,  2,
	18,  0,  0,
	19,  0,  0,
	-1, -1, -1
};

const int PITWALK[27] = {
	18,  0, -1,
	19, -2,  1,
	20, -2,  1,
	21, -2,  1,
	22, -2,  0,
	23, -3,  0,
	24, -3, -1,
	25, -2, -1,
	-1, -1, -1
};

const int PITSTAB[21] = {
	14, -2,   0,
	15, -4,   0,
	16,  3, -13,
	16,  0,   0,
	15, -3,  13,
	14,  4,   0,
	-1, -1,  -1
};

const int TORCH[12] = {
	26, -11, -7,
	27, -12, -2,
	28, -15, -4,
	-1,  -1, -1
};

const int SPEAR[3] = {30, -13, 1};

const int OPENING_OBJS[10][4] = {
	{8, -80, 120, 30},
	{13, 229, 0, 50},
	{12, 78, 0, 50},
	{11, 10, 0, 50},
	{10, 178, 97, 50},
	{9, 92, 192, 50},
	{14, 38, 0, 100},
	{15, 132, 76, 100},
	{16, 142, 0, 100},
	{4, -280, 40, 120},
};

const byte MAP0[26] = {
	0, 0, 0, 1, 1, 1, 0, 0, 1, 0, 4, 0,
	0, 0, 1, 0, 2, 0, 0, 1, 1, 3, 0, 0,
	0, 0xFF
};

const byte MAP1[27] = {
	0, 0, 1, 0, 3, 0, 0, 1, 1, 2, 0, 0,
	0, 0, 1, 1, 1, 0, 0, 0, 1, 0, 4, 0,
	0, 0, 0xFF
};

const byte MAP2[32] = {
	0, 0, 0, 1, 0, 2, 0, 0, 0, 0, 1, 0,
	3, 0, 0, 1, 0, 4, 0, 0, 1, 1, 2, 0,
	0, 1, 0, 1, 0, 0, 0, 0xFF
};

const byte *const MAPTBL[3] = {MAP0, MAP1, MAP2};

const int DOWNRIVEROBJ[14][4] = {
	{ 3, 77, 0, 40 },
	{ 2, 30, 0, 30 },
	{ 2, 290, 0, 50 },
	{ 1, 210, 0, 70 },
	{ 2, 350, 0, 30 },
	{ 1, 370, 0, 20 },
	{ 2, 480, 0, 60 },
	{ 3, 395, 0, 10 },
	{ 1, 550, 0, 30 },
	{ 2, 620, 0, 50 },
	{ 1, 690, 0, 10 },
	{ 2, 715, 0, 40 },
	{ 1, 770, 0, 30 },
	{ 3, 700, 0, 20 }
};

RiverStruct RIVER0OBJECTS[46] = {
	{16,  31, 6400, 0, 4, 12},
	{16,  31, 6200, 0, 2, 12},
	{17,  30, 6100, 0, 3, 15},
	{16,  31, 5970, 0, 7, 12},
	{17,  30, 5910, 0, 5, 15},
	{17,  30, 5730, 0, 3, 15},
	{16,  31, 5700, 0, 7, 12},
	{-1, 314, 5392, 0, 4,  0},
	{17,  30, 5155, 0, 1, 15},
	{16,  31, 5150, 0, 5, 12},
	{16,  31, 5056, 0, 7, 12},
	{17,  30, 4900, 0, 2, 15},
	{17,  30, 4785, 0, 7, 15},
	{16,  31, 4690, 0, 4, 12},
	{16,  31, 4660, 0, 1, 12},
	{17,  30, 4560, 0, 5, 15},
	{16,  31, 4465, 0, 2, 12},
	{-1, 314, 4112, 0, 4,  0},
	{17,  30, 4005, 0, 3, 15},
	{16,  31, 3865, 0, 6, 12},
	{17,  30, 3605, 0, 4, 15},
	{16,  31, 3360, 0, 1, 12},
	{17,  30, 3105, 0, 0, 15},
	{16,  31, 3080, 0, 7, 12},
	{17,  30, 3014, 0, 4, 15},
	{16,  31, 2992, 0, 3, 12},
	{16,  31, 2976, 0, 2, 12},
	{17,  30, 2880, 0, 7, 15},
	{17,  30, 2860, 0, 0, 15},
	{-1, 314, 2512, 0, 4,  0},
	{17,  30, 2270, 0, 4, 15},
	{16,  31, 2195, 0, 6, 12},
	{17,  30, 1824, 0, 1, 15},
	{16,  31, 1776, 0, 4, 12},
	{17,  30, 1650, 0, 3, 15},
	{16,  31, 1616, 0, 7, 12},
	{17,  30, 1585, 0, 2, 15},
	{-1, 314, 1232, 0, 4,  0},
	{17,  30, 1190, 0, 2, 15},
	{16,  31, 1120, 0, 4, 12},
	{17,  30,  970, 0, 7, 15},
	{16,  31,  910, 0, 5, 12},
	{17,  30,  705, 0, 0, 15},
	{16,  31,  550, 0, 4, 12},
	{17,  30,  305, 0, 2, 15},
	{16,  31,  260, 0, 7, 12}
};

RiverStruct RIVER1OBJECTS[50] = {
	{16,  31, 6920, 0, 1, 12},
	{16,  31, 6740, 0, 4, 12},
	{17,  30, 6699, 0, 1, 15},
	{16,  31, 6610, 0, 2, 12},
	{17,  30, 6495, 0, 6, 15},
	{17,  30, 6385, 0, 4, 15},
	{16,  31, 6350, 0, 1, 12},
	{17,  30, 6180, 0, 0, 15},
	{-1, 314, 6032, 0, 4,  0},
	{16,  31, 5800, 0, 3, 12},
	{17,  30, 5790, 0, 6, 15},
	{16,  31, 5530, 0, 4, 12},
	{16,  31, 5500, 0, 7, 12},
	{17,  30, 5495, 0, 1, 15},
	{17,  30, 5376, 0, 0, 15},
	{16,  31, 5328, 0, 7, 12},
	{17,  30, 5248, 0, 2, 15},
	{16,  31, 5248, 0, 6, 12},
	{-1, 314, 4752, 0, 4,  0},
	{17,  30, 4432, 0, 2, 15},
	{16,  31, 4432, 0, 7, 12},
	{16,  31, 4384, 0, 2, 12},
	{17,  30, 4368, 0, 5, 15},
	{16,  31, 4336, 0, 4, 12},
	{17,  30, 4185, 0, 1, 15},
	{16,  31, 4125, 0, 3, 12},
	{17,  30, 3817, 0, 7, 15},
	{16,  31, 3612, 0, 4, 12},
	{16,  31, 3360, 0, 5, 12},
	{16,  31, 3265, 0, 7, 12},
	{17,  30, 3200, 0, 1, 15},
	{17,  30, 3056, 0, 6, 15},
	{-1, 314, 2832, 0, 4,  0},
	{16,  31, 2740, 0, 3, 12},
	{17,  30, 2694, 0, 6, 15},
	{16,  31, 2455, 0, 0, 12},
	{17,  30, 2285, 0, 5, 15},
	{16,  31, 2260, 0, 2, 12},
	{16,  31, 1904, 0, 5, 12},
	{17,  30, 1808, 0, 1, 15},
	{16,  31, 1744, 0, 7, 12},
	{17,  30, 1696, 0, 4, 15},
	{16,  31, 1568, 0, 2, 12},
	{-1, 314, 1232, 0, 4,  0},
	{17,  30,  970, 0, 4, 15},
	{16,  31,  910, 0, 7, 12},
	{17,  30,  705, 0, 0, 15},
	{16,  31,  550, 0, 6, 12},
	{17,  30,  305, 0, 3, 15},
	{ 16, 31, 260, 0, 1, 12 }
};

RiverStruct RIVER2OBJECTS[54] = {
	{16,  31, 8230, 0, 6, 12},
	{16,  31, 8115, 0, 7, 12},
	{17,  30, 7955, 0, 4, 15},
	{16,  31, 7890, 0, 0, 12},
	{16,  31, 7616, 0, 2, 12},
	{17,  30, 7472, 0, 5, 15},
	{16,  31, 7425, 0, 4, 12},
	{17,  30, 7360, 0, 1, 15},
	{16,  31, 7328, 0, 6, 12},
	{-1, 314, 6992, 0, 4,  0},
	{16,  31, 6720, 0, 3, 12},
	{17,  30, 6700, 0, 6, 15},
	{16,  31, 6518, 0, 2, 12},
	{17,  30, 6225, 0, 5, 15},
	{16,  31, 6200, 0, 2, 12},
	{17,  30, 5990, 0, 1, 15},
	{16,  31, 5960, 0, 7, 12},
	{16,  31, 5700, 0, 2, 12},
	{17,  30, 5650, 0, 4, 15},
	{16,  31, 5568, 0, 5, 12},
	{17,  30, 5488, 0, 6, 15},
	{-1, 314, 5072, 0, 4,  0},
	{17,  30, 4825, 0, 4, 15},
	{16,  31, 4782, 0, 2, 12},
	{17,  30, 4660, 0, 5, 15},
	{16,  31, 4510, 0, 7, 12},
	{16,  31, 4495, 0, 1, 12},
	{17,  30, 4250, 0, 2, 15},
	{16,  31, 4195, 0, 4, 12},
	{-1, 314, 3792, 0, 4,  0},
	{17,  30, 3600, 0, 3, 15},
	{16,  31, 3470, 0, 5, 12},
	{16,  31, 3422, 0, 2, 12},
	{17,  30, 3170, 0, 6, 15},
	{16,  31, 2960, 0, 4, 12},
	{17,  30, 2955, 0, 7, 15},
	{-1, 314, 2512, 0, 4,  0},
	{17,  30, 2415, 0, 1, 15},
	{16,  31, 2318, 0, 0, 12},
	{17,  30, 2275, 0, 2, 15},
	{16,  31, 2270, 0, 6, 12},
	{17,  30, 2026, 0, 3, 15},
	{16,  31, 2000, 0, 0, 12},
	{16,  31, 1840, 0, 3, 12},
	{17,  30, 1795, 0, 7, 15},
	{16,  31, 1634, 0, 5, 12},
	{17,  30, 1630, 0, 1, 15},
	{-1, 314, 1232, 0, 4,  0},
	{17,  30,  970, 0, 2, 15},
	{16,  31,  910, 0, 5, 12},
	{17,  30,  705, 0, 0, 15},
	{16,  31,  550, 0, 4, 12},
	{17,  30,  305, 0, 3, 15},
	{16,  31,  260, 0, 6, 12}
};

RiverStruct *RIVER_OBJECTS[3][2] = {
	{ RIVER0OBJECTS, RIVER0OBJECTS + 46 - 1},
	{ RIVER1OBJECTS, RIVER1OBJECTS + 50 - 1 },
	{ RIVER2OBJECTS, RIVER2OBJECTS + 54 - 1 }
};

const int HELP1COORDS[2][4] = {
	{  76, 129, 168, 183 }, { 187, 240, 168, 183 }
};

const int RIVER1OBJ[23][4] = {
	{ 18, -77, 0, 30 },
	{ 18, -325, 0, 20 },
	{ 18, -450, 0, 15 },
	{ 18, -1250, 0, 25 },
	{ 19, -130, 0, 20 },
	{ 19, -410, 0, 15 },
	{ 19, -710, 0, 25 },
	{ 19, -1510, 0, 20 },
	{ 20, -350, 0, 30 },
	{ 20, -695, 0, 25 },
	{ 20, -990, 0, 20 },
	{ 20, -1300, 0, 25 },
	{ 20, -1600, 0, 30 },
	{ 21, -370, 0, 20 },
	{ 21, -650, 0, 30 },
	{ 21, -1215, 0, 40 },
	{ 21, -1815, 0, 35 },
	{ 22, -380, 0, 25 },
	{ 22, -720, 0, 35 },
	{ 22, -1020, 0, 30 },
	{ 22, -1170, 0, 25 },
	{ 22, -1770, 0, 35 },
	{ 23, -500, 63, 20 }
};

const int CAST_END_OBJ[26][4] = {
	{ 0, 118, 210, 10 },
	{ 1, 38, 250, 10 },
	{ 2, 38, 280, 10 },
	{ 3, 38, 310, 10 },
	{ 4, 38, 340, 10 },
	{ 5, 38, 370, 10 },
	{ 6, 38, 400, 10 },
	{ 7, 38, 430, 10 },
	{ 8, 38, 460, 10 },
	{ 9, 38, 490, 10 },
	{ 10, 38, 520, 10 },
	{ 11, 38, 550, 10 },
	{ 12, 38, 580, 10 },
	{ 13, 38, 610, 10 },
	{ 14, 38, 640, 10 },
	{ 15, 38, 670, 10 },
	{ 16, 38, 700, 10 },
	{ 17, 38, 730, 10 },
	{ 18, 38, 760, 10 },
	{ 19, 38, 790, 10 },
	{ 20, 95, 820, 10 },
	{ 21, 94, 850, 10 },
	{ 22, 96, 880, 10 },
	{ 23, 114, 910, 10 },
	{ 24, 114, 940, 10 },
	{ 25, 110, 970, 10 }
};

const int CAST_END_OBJ1[4][4] = {
	{ 0, 40, 1100, 10 },
	{ 2, 11, 1180, 10 },
	{ 1, 154, 1180, 10 },
	{ 3, 103, 1300, 10 }
};


} // End of namespace Amazon
} // End of namespace Access
