/* The FIR coefficients were generated by the following code:

#include <stdio.h>
#include <math.h>

// The Parks-McClellan algorithm implementation is provided by firpm:
// https://github.com/sfilip/firpm
#include <firpm.h>

static const char *get_pm_status_str(pm::status_t status)
{
    switch (status)
    {
#define X(s) \
    case (pm::status_t::s): \
        return #s;

X(STATUS_SUCCESS)
X(STATUS_FREQUENCY_INVALID_INTERVAL)
X(STATUS_AMPLITUDE_VECTOR_MISMATCH)
X(STATUS_AMPLITUDE_DISCONTINUITY)
X(STATUS_WEIGHT_NEGATIVE)
X(STATUS_WEIGHT_VECTOR_MISMATCH)
X(STATUS_WEIGHT_DISCONTINUITY)
X(STATUS_SCALING_INVALID)
X(STATUS_AFP_INVALID)
X(STATUS_COEFFICIENT_SET_INVALID)
X(STATUS_EXCHANGE_FAILURE)
X(STATUS_CONVERGENCE_WARNING)
X(STATUS_UNKNOWN_FAILURE)

#undef X
    }

    return "unknown";
}

int main()
{
    // 80-bit floating point is required for the algorithm to converge.
    typedef long double real;

    int fir_width_shift = 6;
    int fir_step_shift = 7;
    int fir_width = 1 << fir_width_shift;
    int fir_step = 1 << fir_step_shift;
    int fir_len = fir_step * fir_width + 1;

    real sample_freq = 44100.0;
    // The transition band is 4.6 kHz wide, which allows achieving a good
    // stopband rejection given a relatively short 64-tap FIR. It extends
    // 1.55 kHz beyond the Nyquist frequency, which, in the case of upsampling
    // can produce some audible aliasing, but it will be attenuated by at least
    // 17 dB, and it allows us to preserve frequencies below 19 kHz. These
    // values are similar to the ones used by the default quality WASAPI
    // resampler.
    real pass_band = 19000.0;
    real stop_band = 23600.0;

    std::vector<real> bands =
    {
        0.0, pass_band * 2.0 / sample_freq / fir_step,
        stop_band * 2.0 / sample_freq / fir_step, 1.0,
    };
    std::vector<real> gains =
    {
        1.0, 1.0,
        0.0, 0.0,
    };
    std::vector<real> weights =
    {
        1.0,
        1.0,
    };

    // Reduce the order by 2 to add zero padding later. This is required for
    // correct linear interpolation.
    int fir_order = fir_len - 2 - 1;

    pm::pmoutput_t<real> output = pm::firpm<real>(fir_order, bands, gains, weights, 0.01, 8);

    int i;

    fprintf(stderr, "iter %d\n", (int)output.iter);
    fprintf(stderr, "delta %f dB\n", log10((double)output.delta) * 20.0);
    fprintf(stderr, "q %f\n", (double)output.q);
    fprintf(stderr, "status %s\n", get_pm_status_str(output.status));

    printf("static const int fir_len = %d;\n", fir_len);
    printf("static const int fir_step = %d;\n", fir_step);
    printf("static const float fir[] = {");
    for (i = 0; i < fir_len; ++i)
    {
        int index = i - 1;
        real value = 0.0;

        if (0 <= index && index < fir_len - 2)
            value = output.h[index] * fir_step;

        printf("%s%17.10e,", i % 4 ? " " : "\n    ", (double)value);
    }
    printf("\n");
    printf("};\n");
}
*/
static const int fir_len = 8193;
static const int fir_step = 128;
static const float fir[] = {
     0.0000000000e+00, -2.2207222127e-04,  1.4025302043e-05,  1.3443390954e-05,
     1.3064550099e-05,  1.2719058022e-05,  1.2397065958e-05,  1.2091378873e-05,
     1.1809655138e-05,  1.1553153971e-05,  1.1315643963e-05,  1.1095538439e-05,
     1.0894920553e-05,  1.0710291391e-05,  1.0541878290e-05,  1.0389249891e-05,
     1.0250640555e-05,  1.0126524867e-05,  1.0014900175e-05,  9.9161827518e-06,
     9.8287035601e-06,  9.7525195439e-06,  9.6864912668e-06,  9.6302738235e-06,
     9.5831366280e-06,  9.5445156124e-06,  9.5139045177e-06,  9.4906478452e-06,
     9.4743911663e-06,  9.4644314122e-06,  9.4604880509e-06,  9.4618896265e-06,
     9.4683559083e-06,  9.4792809128e-06,  9.4943989207e-06,  9.5131366561e-06,
     9.5352460877e-06,  9.5602016889e-06,  9.5877348599e-06,  9.6173911498e-06,
     9.6488922037e-06,  9.6818266732e-06,  9.7159210593e-06,  9.7508073031e-06,
     9.7861893206e-06,  9.8217617487e-06,  9.8572155084e-06,  9.8922779372e-06,
     9.9266587596e-06,  9.9600981437e-06,  9.9923127057e-06,  1.0023071665e-05,
     1.0052096138e-05,  1.0079171772e-05,  1.0104046194e-05,  1.0126507376e-05,
     1.0146315524e-05,  1.0163282130e-05,  1.0177165791e-05,  1.0187799426e-05,
     1.0194961686e-05,  1.0198485264e-05,  1.0198169785e-05,  1.0193857508e-05,
     1.0185355141e-05,  1.0172518887e-05,  1.0155177067e-05,  1.0133183552e-05,
     1.0106389733e-05,  1.0074657852e-05,  1.0037841224e-05,  9.9958245062e-06,
     9.9484696902e-06,  9.8956688137e-06,  9.8373036561e-06,  9.7732720508e-06,
     9.7034626332e-06,  9.6277901735e-06,  9.5461542772e-06,  9.4584747207e-06,
     9.3646749849e-06,  9.2646765785e-06,  9.1584123535e-06,  9.0458226500e-06,
     8.9268432568e-06,  8.8014294784e-06,  8.6695333690e-06,  8.5311159688e-06,
     8.3861398492e-06,  8.2345820910e-06,  8.0764100816e-06,  7.9116138491e-06,
     7.7401804419e-06,  7.5621008787e-06,  7.3773794892e-06,  7.1860190797e-06,
     6.9880279913e-06,  6.7834270236e-06,  6.5722367483e-06,  6.3544856311e-06,
     6.1302089154e-06,  5.8994474783e-06,  5.6622405342e-06,  5.4186465907e-06,
     5.1687174724e-06,  4.9125167274e-06,  4.6501169096e-06,  4.3815872472e-06,
     4.1070089652e-06,  3.8264677239e-06,  3.5400534208e-06,  3.2478620037e-06,
     2.9499996526e-06,  2.6465707420e-06,  2.3376882590e-06,  2.0234741402e-06,
     1.7040472740e-06,  1.3795403255e-06,  1.0500903114e-06,  7.1583298342e-07,
     3.7691835819e-07,  3.3494103224e-08, -3.1428344793e-07, -6.6625470321e-07,
    -1.0222491308e-06, -1.3821014465e-06, -1.7456307078e-06, -2.1126583213e-06,
    -2.4830013102e-06, -2.8564666919e-06, -3.2328603264e-06, -3.6119853019e-06,
    -3.9936333620e-06, -4.3776009822e-06, -4.7636746501e-06, -5.1516395889e-06,
    -5.5412730743e-06, -5.9323559025e-06, -6.3246534218e-06, -6.7179396637e-06,
    -7.1119784241e-06, -7.5065302494e-06, -7.9013557675e-06, -8.2962095884e-06,
    -8.6908412972e-06, -9.0850034306e-06, -9.4784407932e-06, -9.8708981098e-06,
    -1.0262116982e-05, -1.0651838424e-05, -1.1039794291e-05, -1.1425725372e-05,
    -1.1809360580e-05, -1.2190432878e-05, -1.2568674067e-05, -1.2943810750e-05,
    -1.3315570064e-05, -1.3683679759e-05, -1.4047863365e-05, -1.4407846390e-05,
    -1.4763354528e-05, -1.5114111039e-05, -1.5459837759e-05, -1.5800262576e-05,
    -1.6135103853e-05, -1.6464090073e-05, -1.6786945551e-05, -1.7103395216e-05,
    -1.7413166460e-05, -1.7715987582e-05, -1.8011585395e-05, -1.8299692229e-05,
    -1.8580041036e-05, -1.8852364955e-05, -1.9116401148e-05, -1.9371890077e-05,
    -1.9618568029e-05, -1.9856184304e-05, -2.0084482176e-05, -2.0303213334e-05,
    -2.0512131162e-05, -2.0710992889e-05, -2.0899556515e-05, -2.1077588914e-05,
    -2.1244857054e-05, -2.1401133411e-05, -2.1546197041e-05, -2.1679829163e-05,
    -2.1801814071e-05, -2.1911947638e-05, -2.2010021997e-05, -2.2095842963e-05,
    -2.2169217607e-05, -2.2229960317e-05, -2.2277889205e-05, -2.2312832563e-05,
    -2.2334619810e-05, -2.2343090766e-05, -2.2338092478e-05, -2.2319475175e-05,
    -2.2287099107e-05, -2.2240831923e-05, -2.2180543920e-05, -2.2106119745e-05,
    -2.2017446878e-05, -2.1914422742e-05, -2.1796951838e-05, -2.1664948346e-05,
    -2.1518330538e-05, -2.1357029905e-05, -2.1180984600e-05, -2.0990139418e-05,
    -2.0784452769e-05, -2.0563887334e-05, -2.0328415472e-05, -2.0078021512e-05,
    -1.9812695387e-05, -1.9532438832e-05, -1.9237262803e-05, -1.8927187450e-05,
    -1.8602240308e-05, -1.8262462956e-05, -1.7907902366e-05, -1.7538617258e-05,
    -1.7154678702e-05, -1.6756162746e-05, -1.6343159044e-05, -1.5915766513e-05,
    -1.5474092103e-05, -1.5018255697e-05, -1.4548386487e-05, -1.4064623334e-05,
    -1.3567114672e-05, -1.3056022025e-05, -1.2531511557e-05, -1.1993765251e-05,
    -1.1442973052e-05, -1.0879333648e-05, -1.0303059041e-05, -9.7143687730e-06,
    -9.1134919642e-06, -8.5006701688e-06, -7.8761532193e-06, -7.2402010907e-06,
    -6.5930842526e-06, -5.9350833954e-06, -5.2664849159e-06, -4.5875910177e-06,
    -3.8987079130e-06, -3.2001539672e-06, -2.4922577778e-06, -1.7753544359e-06,
    -1.0497892491e-06, -3.1591717180e-07,  4.2589953198e-07,  1.1752893268e-06,
     1.9318705150e-06,  2.6952553278e-06,  3.4650478950e-06,  4.2408412908e-06,
     5.0222259747e-06,  5.8087799945e-06,  6.6000753094e-06,  7.3956781740e-06,
     8.1951455703e-06,  8.9980289794e-06,  9.8038736906e-06,  1.0612218003e-05,
     1.1422592342e-05,  1.2234524742e-05,  1.3047533992e-05,  1.3861134879e-05,
     1.4674839231e-05,  1.5488149763e-05,  1.6300567469e-05,  1.7111588349e-05,
     1.7920702601e-05,  1.8727398642e-05,  1.9531161072e-05,  2.0331470412e-05,
     2.1127804002e-05,  2.1919638697e-05,  2.2706444615e-05,  2.3487693968e-05,
     2.4262856130e-05,  2.5031397040e-05,  2.5792784957e-05,  2.6546484524e-05,
     2.7291960236e-05,  2.8028677452e-05,  2.8756101610e-05,  2.9473697466e-05,
     3.0180932574e-05,  3.0877274905e-05,  3.1562191751e-05,  3.2235156255e-05,
     3.2895640643e-05,  3.3543120504e-05,  3.4177076653e-05,  3.4796989732e-05,
     3.5402346093e-05,  3.5992635558e-05,  3.6567352371e-05,  3.7125994594e-05,
     3.7668068149e-05,  3.8193081417e-05,  3.8700549342e-05,  3.9189994847e-05,
     3.9660943730e-05,  4.0112931836e-05,  4.0545501813e-05,  4.0958202151e-05,
     4.1350591280e-05,  4.1722234778e-05,  4.2072706450e-05,  4.2401589466e-05,
     4.2708477794e-05,  4.2992972209e-05,  4.3254686068e-05,  4.3493242409e-05,
     4.3708272358e-05,  4.3899422113e-05,  4.4066346065e-05,  4.4208711448e-05,
     4.4326197916e-05,  4.4418496703e-05,  4.4485310640e-05,  4.4526357091e-05,
     4.4541365774e-05,  4.4530078862e-05,  4.4492254989e-05,  4.4427664265e-05,
     4.4336090909e-05,  4.4217336275e-05,  4.4071212702e-05,  4.3897550682e-05,
     4.3696194971e-05,  4.3467005736e-05,  4.3209858183e-05,  4.2924645168e-05,
     4.2611273480e-05,  4.2269667296e-05,  4.1899769023e-05,  4.1501534614e-05,
     4.1074939493e-05,  4.0619975608e-05,  4.0136650171e-05,  3.9624990596e-05,
     3.9085040356e-05,  3.8516860729e-05,  3.7920531048e-05,  3.7296149060e-05,
     3.6643828359e-05,  3.5963702805e-05,  3.5255924261e-05,  3.4520660669e-05,
     3.3758101977e-05,  3.2968453258e-05,  3.2151938700e-05,  3.1308801939e-05,
     3.0439303782e-05,  2.9543723858e-05,  2.8622361050e-05,  2.7675531988e-05,
     2.6703570458e-05,  2.5706830983e-05,  2.4685684116e-05,  2.3640519043e-05,
     2.2571745344e-05,  2.1479787202e-05,  2.0365089055e-05,  1.9228112530e-05,
     1.8069336025e-05,  1.6889256404e-05,  1.5688388352e-05,  1.4467262871e-05,
     1.3226427878e-05,  1.1966449967e-05,  1.0687908949e-05,  9.3914041685e-06,
     8.0775504331e-06,  6.7469772257e-06,  5.4003319185e-06,  4.0382759202e-06,
     2.6614854151e-06,  1.2706526834e-06, -1.3351574843e-07, -1.5502995596e-06,
    -2.9789635555e-06, -4.4187585519e-06, -5.8689247276e-06, -7.3286854427e-06,
    -8.7972545674e-06, -1.0273832441e-05, -1.1757606636e-05, -1.3247754791e-05,
    -1.4743443094e-05, -1.6243826513e-05, -1.7748050524e-05, -1.9255250978e-05,
    -2.0764552764e-05, -2.2275073465e-05, -2.3785922435e-05, -2.5296198977e-05,
    -2.6804998254e-05, -2.8311405748e-05, -2.9814500967e-05, -3.1313358448e-05,
    -3.2807046129e-05, -3.4294627479e-05, -3.5775162091e-05, -3.7247705532e-05,
    -3.8711308545e-05, -4.0165021644e-05, -4.1607890817e-05, -4.3038961104e-05,
    -4.4457278158e-05, -4.5861884526e-05, -4.7251824498e-05, -4.8626142807e-05,
    -4.9983884414e-05, -5.1324097037e-05, -5.2645831178e-05, -5.3948139655e-05,
    -5.5230078745e-05, -5.6490710627e-05, -5.7729099321e-05, -5.8944316859e-05,
    -6.0135440957e-05, -6.1301554543e-05, -6.2441750112e-05, -6.3555126423e-05,
    -6.4640790833e-05, -6.5697860266e-05, -6.6725461804e-05, -6.7722731611e-05,
    -6.8688818451e-05, -6.9622882176e-05, -7.0524093207e-05, -7.1391637335e-05,
    -7.2224711884e-05, -7.3022528338e-05, -7.3784314572e-05, -7.4509311404e-05,
    -7.5196776589e-05, -7.5845983915e-05, -7.6456224337e-05, -7.7026805187e-05,
    -7.7557054282e-05, -7.8046315827e-05, -7.8493953716e-05, -7.8899352834e-05,
    -7.9261915930e-05, -7.9581068301e-05, -7.9856256706e-05, -8.0086948372e-05,
    -8.0272634128e-05, -8.0412827062e-05, -8.0507063173e-05, -8.0554902160e-05,
    -8.0555929494e-05, -8.0509752923e-05, -8.0416007310e-05, -8.0274352648e-05,
    -8.0084473067e-05, -7.9846081556e-05, -7.9558916173e-05, -7.9222742486e-05,
    -7.8837354073e-05, -7.8402572034e-05, -7.7918244723e-05, -7.7384250050e-05,
    -7.6800494495e-05, -7.6166912393e-05, -7.5483469750e-05, -7.4750160337e-05,
    -7.3967007607e-05, -7.3134066746e-05, -7.2251420923e-05, -7.1319185350e-05,
    -7.0337505652e-05, -6.9306558094e-05, -6.8226549238e-05, -6.7097717989e-05,
    -6.5920332937e-05, -6.4694694169e-05, -6.3421134747e-05, -6.2100016584e-05,
    -6.0731734831e-05, -5.9316715619e-05, -5.7855415230e-05, -5.6348322743e-05,
    -5.4795958094e-05, -5.3198872177e-05, -5.1557647345e-05, -4.9872897489e-05,
    -4.8145265822e-05, -4.6375427623e-05, -4.4564089070e-05, -4.2711984866e-05,
    -4.0819882795e-05, -3.8888578380e-05, -3.6918897431e-05, -3.4911695691e-05,
    -3.2867857702e-05, -3.0788296645e-05, -2.8673955273e-05, -2.6525804074e-05,
    -2.4344840469e-05, -2.2132091046e-05, -1.9888608087e-05, -1.7615470449e-05,
    -1.5313785194e-05, -1.2984682608e-05, -1.0629319914e-05, -8.2488786137e-06,
    -5.8445640543e-06, -3.4176054407e-06, -9.6925600492e-07,  1.4992093827e-06,
     3.9864938426e-06,  6.4912777663e-06,  9.0122232304e-06,  1.1547970285e-05,
     1.4097139687e-05,  1.6658334627e-05,  1.9230138559e-05,  2.1811118542e-05,
     2.4399825308e-05,  2.6994793179e-05,  2.9594541113e-05,  3.2197574606e-05,
     3.4802384587e-05,  3.7407449042e-05,  4.0011235977e-05,  4.2612199662e-05,
     4.5208786033e-05,  4.7799431087e-05,  5.0382561297e-05,  5.2956596559e-05,
     5.5519949790e-05,  5.8071027665e-05,  6.0608232118e-05,  6.3129961696e-05,
     6.5634610103e-05,  6.8120570241e-05,  7.0586233853e-05,  7.3029990912e-05,
     7.5450234310e-05,  7.7845357071e-05,  8.0213754755e-05,  8.2553827450e-05,
     8.4863979018e-05,  8.7142619126e-05,  8.9388164368e-05,  9.1599039209e-05,
     9.3773675049e-05,  9.5910515158e-05,  9.8008011468e-05,  1.0006462804e-04,
     1.0207884270e-04,  1.0404914536e-04,  1.0597404128e-04,  1.0785205158e-04,
     1.0968171311e-04,  1.1146158087e-04,  1.1319022886e-04,  1.1486625035e-04,
     1.1648825868e-04,  1.1805489052e-04,  1.1956480249e-04,  1.2101667672e-04,
     1.2240921955e-04,  1.2374116184e-04,  1.2501126220e-04,  1.2621830587e-04,
     1.2736110603e-04,  1.2843850543e-04,  1.2944937729e-04,  1.3039262487e-04,
     1.3126718433e-04,  1.3207202467e-04,  1.3280614684e-04,  1.3346858841e-04,
     1.3405842073e-04,  1.3457475144e-04,  1.3501672622e-04,  1.3538352724e-04,
     1.3567437549e-04,  1.3588853127e-04,  1.3602529482e-04,  1.3608400615e-04,
     1.3606404842e-04,  1.3596484537e-04,  1.3578586342e-04,  1.3552661361e-04,
     1.3518664883e-04,  1.3476556783e-04,  1.3426301445e-04,  1.3367867743e-04,
     1.3301229212e-04,  1.3226364062e-04,  1.3143255142e-04,  1.3051890057e-04,
     1.2952261325e-04,  1.2844366126e-04,  1.2728206638e-04,  1.2603789957e-04,
     1.2471127938e-04,  1.2330237608e-04,  1.2181140851e-04,  1.2023864604e-04,
     1.1858440854e-04,  1.1684906667e-04,  1.1503304057e-04,  1.1313680225e-04,
     1.1116087430e-04,  1.0910582933e-04,  1.0697229253e-04,  1.0476093901e-04,
     1.0247249424e-04,  1.0010773608e-04,  9.7667491240e-05,  9.5152638223e-05,
     9.2564105725e-05,  8.9902872945e-05,  8.7169968110e-05,  8.4366470716e-05,
     8.1493508276e-05,  7.8552257741e-05,  7.5543946099e-05,  7.2469847095e-05,
     6.9331283543e-05,  6.6129625953e-05,  6.2866290692e-05,  5.9542741834e-05,
     5.6160489179e-05,  5.2721087922e-05,  4.9226138010e-05,  4.5677284562e-05,
     4.2076214553e-05,  3.8424659042e-05,  3.4724391359e-05,  3.0977224886e-05,
     2.7185015534e-05,  2.3349657761e-05,  1.9473084947e-05,  1.5557269073e-05,
     1.1604218952e-05,  7.6159794021e-06,  3.5946311355e-06, -4.5771077505e-07,
    -4.5388992208e-06, -8.6467531740e-06, -1.2779061749e-05, -1.6933583969e-05,
    -2.1108048569e-05, -2.5300158002e-05, -2.9507587299e-05, -3.3727986033e-05,
    -3.7958980084e-05, -4.2198172267e-05, -4.6443143046e-05, -5.0691452928e-05,
    -5.4940643700e-05, -5.9188237820e-05, -6.3431743560e-05, -6.7668652465e-05,
    -7.1896442656e-05, -7.6112580587e-05, -8.0314520972e-05, -8.4499709279e-05,
    -8.8665583629e-05, -9.2809575053e-05, -9.6929109405e-05, -1.0102160957e-04,
    -1.0508449585e-04, -1.0911518740e-04, -1.1311110633e-04, -1.1706967496e-04,
    -1.2098832136e-04, -1.2486447867e-04, -1.2869558673e-04, -1.3247909448e-04,
    -1.3621246153e-04, -1.3989315871e-04, -1.4351867074e-04, -1.4708649785e-04,
    -1.5059415568e-04, -1.5403917882e-04, -1.5741912217e-04, -1.6073156044e-04,
    -1.6397409334e-04, -1.6714434382e-04, -1.7023996109e-04, -1.7325862237e-04,
    -1.7619803400e-04, -1.7905593277e-04, -1.8183008851e-04, -1.8451830503e-04,
    -1.8711842038e-04, -1.8962831118e-04, -1.9204589133e-04, -1.9436911463e-04,
    -1.9659597763e-04, -1.9872451840e-04, -2.0075282008e-04, -2.0267901162e-04,
    -2.0450126880e-04, -2.0621781588e-04, -2.0782692795e-04, -2.0932693078e-04,
    -2.1071620267e-04, -2.1199317712e-04, -2.1315634111e-04, -2.1420423930e-04,
    -2.1513547432e-04, -2.1594870681e-04, -2.1664265879e-04, -2.1721611305e-04,
    -2.1766791479e-04, -2.1799697265e-04, -2.1820226079e-04, -2.1828281774e-04,
    -2.1823774998e-04, -2.1806623153e-04, -2.1776750393e-04, -2.1734087960e-04,
    -2.1678574064e-04, -2.1610154002e-04, -2.1528780398e-04, -2.1434413054e-04,
    -2.1327019159e-04, -2.1206573307e-04, -2.1073057622e-04, -2.0926461644e-04,
    -2.0766782704e-04, -2.0594025652e-04, -2.0408203058e-04, -2.0209335315e-04,
    -1.9997450482e-04, -1.9772584489e-04, -1.9534781161e-04, -1.9284092126e-04,
    -1.9020576955e-04, -1.8744303139e-04, -1.8455346064e-04, -1.8153789008e-04,
    -1.7839723329e-04, -1.7513248147e-04, -1.7174470638e-04, -1.6823505891e-04,
    -1.6460476788e-04, -1.6085514210e-04, -1.5698756837e-04, -1.5300351167e-04,
    -1.4890451527e-04, -1.4469220014e-04, -1.4036826345e-04, -1.3593447959e-04,
    -1.3139269913e-04, -1.2674484690e-04, -1.2199292422e-04, -1.1713900550e-04,
    -1.1218523829e-04, -1.0713384367e-04, -1.0198711337e-04, -9.6747410415e-05,
    -9.1417167945e-05, -8.5998888071e-05, -8.0495140147e-05, -7.4908561415e-05,
    -6.9241853969e-05, -6.3497784196e-05, -5.7679183265e-05, -5.1788942772e-05,
    -4.5830016186e-05, -3.9805416301e-05, -3.3718213322e-05, -2.7571534424e-05,
    -2.1368562197e-05, -1.5112532384e-05, -8.8067329427e-06, -2.4545029361e-06,
     3.9407709467e-06,  1.0375654283e-05,  1.6846666687e-05,  2.3350285625e-05,
     2.9882944837e-05,  3.6441039183e-05,  4.3020925141e-05,  4.9618922894e-05,
     5.6231318031e-05,  6.2854364520e-05,  6.9484285038e-05,  7.6117274032e-05,
     8.2749500508e-05,  8.9377108135e-05,  9.5996219170e-05,  1.0260293639e-04,
     1.0919334330e-04,  1.1576350924e-04,  1.2230948936e-04,  1.2882732788e-04,
     1.3531306011e-04,  1.4176271530e-04,  1.4817231729e-04,  1.5453788866e-04,
     1.6085545240e-04,  1.6712103305e-04,  1.7333066172e-04,  1.7948037603e-04,
     1.8556622312e-04,  1.9158426308e-04,  1.9753056967e-04,  2.0340123381e-04,
     2.0919236604e-04,  2.1490009865e-04,  2.2052058744e-04,  2.2605001573e-04,
     2.3148459507e-04,  2.3682056803e-04,  2.4205421235e-04,  2.4718184044e-04,
     2.5219980422e-04,  2.5710449653e-04,  2.6189235293e-04,  2.6655985499e-04,
     2.7110353255e-04,  2.7551996555e-04,  2.7980578653e-04,  2.8395768397e-04,
     2.8797240223e-04,  2.9184674612e-04,  2.9557758242e-04,  2.9916184095e-04,
     3.0259651903e-04,  3.0587868188e-04,  3.0900546494e-04,  3.1197407688e-04,
     3.1478180088e-04,  3.1742599680e-04,  3.1990410359e-04,  3.2221364166e-04,
     3.2435221276e-04,  3.2631750507e-04,  3.2810729216e-04,  3.2971943622e-04,
     3.3115189045e-04,  3.3240269940e-04,  3.3347000150e-04,  3.3435203099e-04,
     3.3504711859e-04,  3.3555369363e-04,  3.3587028611e-04,  3.3599552744e-04,
     3.3592815151e-04,  3.3566699814e-04,  3.3521101103e-04,  3.3455924205e-04,
     3.3371085122e-04,  3.3266510744e-04,  3.3142139069e-04,  3.2997919248e-04,
     3.2833811638e-04,  3.2649787950e-04,  3.2445831381e-04,  3.2221936545e-04,
     3.1978109710e-04,  3.1714368832e-04,  3.1430743444e-04,  3.1127274985e-04,
     3.0804016627e-04,  3.0461033387e-04,  3.0098402235e-04,  2.9716212026e-04,
     2.9314563533e-04,  2.8893569501e-04,  2.8453354650e-04,  2.7994055557e-04,
     2.7515820890e-04,  2.7018811170e-04,  2.6503198811e-04,  2.5969168214e-04,
     2.5416915503e-04,  2.4846648662e-04,  2.4258587451e-04,  2.3652963300e-04,
     2.3030019265e-04,  2.2390010016e-04,  2.1733201645e-04,  2.1059871618e-04,
     2.0370308806e-04,  1.9664813160e-04,  1.8943695792e-04,  1.8207278831e-04,
     1.7455895136e-04,  1.6689888405e-04,  1.5909612860e-04,  1.5115433186e-04,
     1.4307724350e-04,  1.3486871525e-04,  1.2653269742e-04,  1.1807323909e-04,
     1.0949448556e-04,  1.0080067566e-04,  9.1996141801e-05,  8.3085306462e-05,
     7.4072680009e-05,  6.4962860155e-05,  5.5760527745e-05,  4.6470445978e-05,
     3.7097457591e-05,  2.7646483029e-05,  1.8122516553e-05,  8.5306260705e-06,
    -1.1240516679e-06, -1.0836312746e-05, -2.0600887678e-05, -3.0412446111e-05,
    -4.0265598031e-05, -5.0154896790e-05, -6.0074842988e-05, -7.0019886250e-05,
    -7.9984428447e-05, -8.9962826915e-05, -9.9949397617e-05, -1.0993841696e-04,
    -1.1992412724e-04, -1.2990073779e-04, -1.3986242857e-04, -1.4980335472e-04,
    -1.5971764759e-04, -1.6959941975e-04, -1.7944276796e-04, -1.8924177616e-04,
    -1.9899051896e-04, -2.0868306563e-04, -2.1831348259e-04, -2.2787583696e-04,
    -2.3736420153e-04, -2.4677265582e-04, -2.5609529166e-04, -2.6532621616e-04,
    -2.7445955404e-04, -2.8348945291e-04, -2.9241008579e-04, -3.0121565471e-04,
    -3.0990039469e-04, -3.1845857742e-04, -3.2688451378e-04, -3.3517255837e-04,
    -3.4331711326e-04, -3.5131262971e-04, -3.5915361467e-04, -3.6683463173e-04,
    -3.7435030563e-04, -3.8169532607e-04, -3.8886445050e-04, -3.9585250762e-04,
    -4.0265440165e-04, -4.0926511502e-04, -4.1567971150e-04, -4.2189334058e-04,
    -4.2790123987e-04, -4.3369873796e-04, -4.3928125994e-04, -4.4464432756e-04,
    -4.4978356489e-04, -4.5469470034e-04, -4.5937356967e-04, -4.6381611914e-04,
    -4.6801840920e-04, -4.7197661643e-04, -4.7568703692e-04, -4.7914608985e-04,
    -4.8235031850e-04, -4.8529639445e-04, -4.8798112030e-04, -4.9040143070e-04,
    -4.9255439719e-04, -4.9443722903e-04, -4.9604727600e-04, -4.9738203094e-04,
    -4.9843913217e-04, -4.9921636488e-04, -4.9971166435e-04, -4.9992311786e-04,
    -4.9984896527e-04, -4.9948760314e-04, -4.9883758472e-04, -4.9789762209e-04,
    -4.9666658899e-04, -4.9514352080e-04, -4.9332761696e-04, -4.9121824226e-04,
    -4.8881492793e-04, -4.8611737247e-04, -4.8312544406e-04, -4.7983918041e-04,
    -4.7625878973e-04, -4.7238465307e-04, -4.6821732264e-04, -4.6375752434e-04,
    -4.5900615800e-04, -4.5396429690e-04, -4.4863318945e-04, -4.4301425869e-04,
    -4.3710910244e-04, -4.3091949317e-04, -4.2444737913e-04, -4.1769488225e-04,
    -4.1066429974e-04, -4.0335810319e-04, -3.9577893697e-04, -3.8792961948e-04,
    -3.7981314125e-04, -3.7143266413e-04, -3.6279152153e-04, -3.5389321637e-04,
    -3.4474142031e-04, -3.3533997254e-04, -3.2569287905e-04, -3.1580430945e-04,
    -3.0567859821e-04, -2.9532024073e-04, -2.8473389243e-04, -2.7392436768e-04,
    -2.6289663637e-04, -2.5165582286e-04, -2.4020720405e-04, -2.2855620650e-04,
    -2.1670840439e-04, -2.0466951749e-04, -1.9244540802e-04, -1.8004207773e-04,
    -1.6746566695e-04, -1.5472244930e-04, -1.4181883079e-04, -1.2876134633e-04,
    -1.1555665558e-04, -1.0221154121e-04, -8.8732904839e-05, -7.5127763719e-05,
    -6.1403247458e-05, -4.7566595062e-05, -3.3625149860e-05, -1.9586357133e-05,
    -5.4577601933e-06,  8.7530047847e-06,  2.3038210511e-05,  3.7390044160e-05,
     5.1800611318e-05,  6.6261939179e-05,  8.0765981890e-05,  9.5304624323e-05,
     1.0986968624e-04,  1.2445292656e-04,  1.3904604891e-04,  1.5364070439e-04,
     1.6822849768e-04,  1.8280099146e-04,  1.9734970968e-04,  2.1186614429e-04,
     2.2634175852e-04,  2.4076799206e-04,  2.5513626642e-04,  2.6943798948e-04,
     2.8366455997e-04,  2.9780737315e-04,  3.1185782561e-04,  3.2580731928e-04,
     3.3964726830e-04,  3.5336910278e-04,  3.6696427364e-04,  3.8042425934e-04,
     3.9374056875e-04,  4.0690474787e-04,  4.1990838437e-04,  4.3274311293e-04,
     4.4540061976e-04,  4.5787264891e-04,  4.7015100607e-04,  4.8222756395e-04,
     4.9409426852e-04,  5.0574314230e-04,  5.1716629071e-04,  5.2835590720e-04,
     5.3930427680e-04,  5.5000378278e-04,  5.6044691062e-04,  5.7062625317e-04,
     5.8053451546e-04,  5.9016452035e-04,  5.9950921170e-04,  6.0856166052e-04,
     6.1731506951e-04,  6.2576277638e-04,  6.3389826063e-04,  6.4171514648e-04,
     6.4920720766e-04,  6.5636837241e-04,  6.6319272717e-04,  6.6967452088e-04,
     6.7580816979e-04,  6.8158826136e-04,  6.8700955758e-04,  6.9206700039e-04,
     6.9675571433e-04,  7.0107101024e-04,  7.0500839068e-04,  7.0856355135e-04,
     7.1173238622e-04,  7.1451099051e-04,  7.1689566370e-04,  7.1888291285e-04,
     7.2046945635e-04,  7.2165222622e-04,  7.2242837106e-04,  7.2279526010e-04,
     7.2275048380e-04,  7.2229185800e-04,  7.2141742626e-04,  7.2012546101e-04,
     7.1841446745e-04,  7.1628318466e-04,  7.1373058754e-04,  7.1075588895e-04,
     7.0735854158e-04,  7.0353823882e-04,  6.9929491691e-04,  6.9462875667e-04,
     6.8954018290e-04,  6.8402986779e-04,  6.7809872995e-04,  6.7174793573e-04,
     6.6497890056e-04,  6.5779328861e-04,  6.5019301330e-04,  6.4218023795e-04,
     6.3375737538e-04,  6.2492708750e-04,  6.1569228616e-04,  6.0605613203e-04,
     5.9602203359e-04,  5.8559364831e-04,  5.7477487937e-04,  5.6356987642e-04,
     5.5198303409e-04,  5.4001899024e-04,  5.2768262511e-04,  5.1497905976e-04,
     5.0191365396e-04,  4.8849200422e-04,  4.7471994286e-04,  4.6060353421e-04,
     4.4614907367e-04,  4.3136308522e-04,  4.1625231747e-04,  4.0082374267e-04,
     3.8508455272e-04,  3.6904215616e-04,  3.5270417572e-04,  3.3607844463e-04,
     3.1917300287e-04,  3.0199609396e-04,  2.8455616156e-04,  2.6686184416e-04,
     2.4892197325e-04,  2.3074556760e-04,  2.1234182925e-04,  1.9372014013e-04,
     1.7489005610e-04,  1.5586130315e-04,  1.3664377270e-04,  1.1724751650e-04,
     9.7682741408e-05,  7.7959805110e-05,  5.8089210022e-05,  3.8081597810e-05,
     1.7947745394e-05, -2.3014425533e-06, -2.2654937848e-05, -4.3101594603e-05,
    -6.3630156232e-05, -8.4229260112e-05, -1.0488744432e-04, -1.2559315357e-04,
    -1.4633474568e-04, -1.6710049680e-04, -1.8787860954e-04, -2.0865721803e-04,
    -2.2942439465e-04, -2.5016815772e-04, -2.7087647648e-04, -2.9153727876e-04,
    -3.1213845808e-04, -3.3266787939e-04, -3.5311338703e-04, -3.7346281121e-04,
    -3.9370397501e-04, -4.1382470087e-04, -4.3381281907e-04, -4.5365617287e-04,
    -4.7334262719e-04, -4.9286007553e-04, -5.1219644595e-04, -5.3133970976e-04,
    -5.5027788777e-04, -5.6899905745e-04, -5.8749136069e-04, -6.0574301083e-04,
    -6.2374229932e-04, -6.4147760348e-04, -6.5893739377e-04, -6.7611023969e-04,
    -6.9298481879e-04, -7.0954992251e-04, -7.2579446313e-04, -7.4170748209e-04,
    -7.5727815535e-04, -7.7249580127e-04, -7.8734988744e-04, -8.0183003753e-04,
    -8.1592603752e-04, -8.2962784361e-04, -8.4292558776e-04, -8.5580958431e-04,
    -8.6827033779e-04, -8.8029854759e-04, -8.9188511557e-04, -9.0302115239e-04,
    -9.1369798262e-04, -9.2390715186e-04, -9.3364043238e-04, -9.4288982889e-04,
    -9.5164758437e-04, -9.5990618650e-04, -9.6765837187e-04, -9.7489713238e-04,
    -9.8161572081e-04, -9.8780765453e-04, -9.9346672261e-04, -9.9858698940e-04,
    -1.0031627996e-03, -1.0071887833e-03, -1.0106598603e-03, -1.0135712440e-03,
    -1.0159184465e-03, -1.0176972823e-03, -1.0189038719e-03, -1.0195346464e-03,
    -1.0195863503e-03, -1.0190560448e-03, -1.0179411125e-03, -1.0162392587e-03,
    -1.0139485154e-03, -1.0110672442e-03, -1.0075941383e-03, -1.0035282247e-03,
    -9.9886886755e-04, -9.9361576900e-04, -9.8776897145e-04, -9.8132886010e-04,
    -9.7429616279e-04, -9.6667195237e-04, -9.5845764794e-04, -9.4965501455e-04,
    -9.4026616535e-04, -9.3029356132e-04, -9.1974001157e-04, -9.0860867349e-04,
    -8.9690305305e-04, -8.8462700360e-04, -8.7178472613e-04, -8.5838076890e-04,
    -8.4442002494e-04, -8.2990773292e-04, -8.1484947439e-04, -7.9925117247e-04,
    -7.8311909082e-04, -7.6645983103e-04, -7.4928033057e-04, -7.3158786075e-04,
    -7.1339002400e-04, -6.9469475045e-04, -6.7551029620e-04, -6.5584523934e-04,
    -6.3570847623e-04, -6.1510921948e-04, -5.9405699218e-04, -5.7256162515e-04,
    -5.5063325270e-04, -5.2828230771e-04, -5.0551951750e-04, -4.8235589924e-04,
    -4.5880275462e-04, -4.3487166439e-04, -4.1057448433e-04, -3.8592333797e-04,
    -3.6093061226e-04, -3.3560895164e-04, -3.0997125085e-04, -2.8403065003e-04,
    -2.5780052767e-04, -2.3129449377e-04, -2.0452638380e-04, -1.7751025159e-04,
    -1.5026036191e-04, -1.2279118354e-04, -9.5117382422e-05, -6.7253812496e-05,
    -3.9215510220e-05, -1.1017685271e-05,  1.7324287146e-05,  4.5794873620e-05,
     7.4378391705e-05,  1.0305901794e-04,  1.3182079596e-04,  1.6064764531e-04,
     1.8952337045e-04,  2.1843166870e-04,  2.4735613982e-04,  2.7628029540e-04,
     3.0518756644e-04,  3.3406131433e-04,  3.6288483893e-04,  3.9164138788e-04,
     4.2031416694e-04,  4.4888634864e-04,  4.7734108213e-04,  5.0566150281e-04,
     5.3383074225e-04,  5.6183193681e-04,  5.8964823902e-04,  6.1726282621e-04,
     6.4465891037e-04,  6.7181974913e-04,  6.9872865413e-04,  7.2536900169e-04,
     7.5172424294e-04,  7.7777791298e-04,  8.0351364139e-04,  8.2891516210e-04,
     8.5396632309e-04,  8.7865109578e-04,  9.0295358622e-04,  9.2685804314e-04,
     9.5034886901e-04,  9.7341062977e-04,  9.9602806338e-04,  1.0181860908e-03,
     1.0398698246e-03,  1.0610645788e-03,  1.0817558784e-03,  1.1019294686e-03,
     1.1215713242e-03,  1.1406676584e-03,  1.1592049327e-03,  1.1771698644e-03,
     1.1945494374e-03,  1.2113309095e-03,  1.2275018213e-03,  1.2430500058e-03,
     1.2579635951e-03,  1.2722310299e-03,  1.2858410671e-03,  1.2987827881e-03,
     1.3110456061e-03,  1.3226192747e-03,  1.3334938945e-03,  1.3436599204e-03,
     1.3531081699e-03,  1.3618298289e-03,  1.3698164588e-03,  1.3770600040e-03,
     1.3835527970e-03,  1.3892875657e-03,  1.3942574388e-03,  1.3984559520e-03,
     1.4018770533e-03,  1.4045151094e-03,  1.4063649095e-03,  1.4074216710e-03,
     1.4076810449e-03,  1.4071391185e-03,  1.4057924220e-03,  1.4036379309e-03,
     1.4006730705e-03,  1.3968957194e-03,  1.3923042129e-03,  1.3868973457e-03,
     1.3806743751e-03,  1.3736350241e-03,  1.3657794824e-03,  1.3571084098e-03,
     1.3476229374e-03,  1.3373246688e-03,  1.3262156823e-03,  1.3142985311e-03,
     1.3015762443e-03,  1.2880523277e-03,  1.2737307637e-03,  1.2586160108e-03,
     1.2427130042e-03,  1.2260271544e-03,  1.2085643462e-03,  1.1903309389e-03,
     1.1713337628e-03,  1.1515801187e-03,  1.1310777761e-03,  1.1098349696e-03,
     1.0878603978e-03,  1.0651632197e-03,  1.0417530513e-03,  1.0176399627e-03,
     9.9283447446e-04,  9.6734755292e-04,  9.4119060641e-04,  9.1437548137e-04,
     8.8691445590e-04,  8.5882023657e-04,  8.3010595176e-04,  8.0078514626e-04,
     7.7087177592e-04,  7.4038020112e-04,  7.0932518024e-04,  6.7772186339e-04,
     6.4558578542e-04,  6.1293285799e-04,  5.7977936328e-04,  5.4614194574e-04,
     5.1203760367e-04,  4.7748368256e-04,  4.4249786497e-04,  4.0709816285e-04,
     3.7130290866e-04,  3.3513074590e-04,  2.9860061997e-04,  2.6173176903e-04,
     2.2454371381e-04,  1.8705624735e-04,  1.4928942609e-04,  1.1126355790e-04,
     7.2999192630e-05,  3.4517111558e-05, -4.1616846182e-06, -4.3015985781e-05,
    -8.2024384107e-05, -1.2116528576e-04, -1.6041692201e-04, -1.9975736092e-04,
    -2.3916451970e-04, -2.7861617646e-04, -3.1808998183e-04, -3.5756347282e-04,
    -3.9701408329e-04, -4.3641915772e-04, -4.7575596367e-04, -5.1500170389e-04,
    -5.5413352983e-04, -5.9312855451e-04, -6.3196386511e-04, -6.7061653618e-04,
    -7.0906364328e-04, -7.4728227517e-04, -7.8524954777e-04, -8.2294261786e-04,
    -8.6033869498e-04, -8.9741505643e-04, -9.3414905964e-04, -9.7051815544e-04,
    -1.0064999022e-03, -1.0420719789e-03, -1.0772121980e-03, -1.1118985198e-03,
    -1.1461090651e-03, -1.1798221283e-03, -1.2130161915e-03, -1.2456699371e-03,
    -1.2777622606e-03, -1.3092722853e-03, -1.3401793734e-03, -1.3704631398e-03,
    -1.4001034656e-03, -1.4290805094e-03, -1.4573747212e-03, -1.4849668546e-03,
    -1.5118379793e-03, -1.5379694924e-03, -1.5633431327e-03, -1.5879409904e-03,
    -1.6117455204e-03, -1.6347395539e-03, -1.6569063093e-03, -1.6782294043e-03,
    -1.6986928667e-03, -1.7182811455e-03, -1.7369791216e-03, -1.7547721188e-03,
    -1.7716459139e-03, -1.7875867466e-03, -1.8025813309e-03, -1.8166168623e-03,
    -1.8296810299e-03, -1.8417620243e-03, -1.8528485466e-03, -1.8629298180e-03,
    -1.8719955874e-03, -1.8800361400e-03, -1.8870423053e-03, -1.8930054651e-03,
    -1.8979175599e-03, -1.9017710974e-03, -1.9045591583e-03, -1.9062754027e-03,
    -1.9069140777e-03, -1.9064700213e-03, -1.9049386694e-03, -1.9023160612e-03,
    -1.8985988429e-03, -1.8937842734e-03, -1.8878702281e-03, -1.8808552032e-03,
    -1.8727383183e-03, -1.8635193217e-03, -1.8531985910e-03, -1.8417771369e-03,
    -1.8292566059e-03, -1.8156392803e-03, -1.8009280817e-03, -1.7851265710e-03,
    -1.7682389495e-03, -1.7502700593e-03, -1.7312253833e-03, -1.7111110446e-03,
    -1.6899338060e-03, -1.6677010697e-03, -1.6444208741e-03, -1.6201018936e-03,
    -1.5947534359e-03, -1.5683854387e-03, -1.5410084680e-03, -1.5126337137e-03,
    -1.4832729865e-03, -1.4529387137e-03, -1.4216439350e-03, -1.3894022967e-03,
    -1.3562280480e-03, -1.3221360342e-03, -1.2871416911e-03, -1.2512610396e-03,
    -1.2145106775e-03, -1.1769077736e-03, -1.1384700602e-03, -1.0992158249e-03,
    -1.0591639026e-03, -1.0183336678e-03, -9.7674502498e-04, -9.3441839951e-04,
    -8.9137472902e-04, -8.4763545244e-04, -8.0322250046e-04, -7.5815828566e-04,
    -7.1246569022e-04, -6.6616805643e-04, -6.1928917434e-04, -5.7185326998e-04,
    -5.2388499381e-04, -4.7540940828e-04, -4.2645197487e-04, -3.7703854163e-04,
    -3.2719533012e-04, -2.7694892099e-04, -2.2632624146e-04, -1.7535455085e-04,
    -1.2406142581e-04, -7.2474747267e-05, -2.0622684252e-05,  3.1466320309e-05,
     8.3763564741e-05,  1.3624010420e-04,  1.8886676631e-04,  2.4161416647e-04,
     2.9445272408e-04,  3.4735267927e-04,  4.0028410785e-04,  4.5321693934e-04,
     5.0612097268e-04,  5.5896589280e-04,  6.1172128844e-04,  6.6435666842e-04,
     7.1684147899e-04,  7.6914512146e-04,  8.2123696931e-04,  8.7308638528e-04,
     9.2466273960e-04,  9.7593542731e-04,  1.0268738853e-03,  1.0774476116e-03,
     1.1276261816e-03,  1.1773792664e-03,  1.2266766510e-03,  1.2754882514e-03,
     1.3237841330e-03,  1.3715345284e-03,  1.4187098549e-03,  1.4652807321e-03,
     1.5112180008e-03,  1.5564927388e-03,  1.6010762799e-03,  1.6449402315e-03,
     1.6880564908e-03,  1.7303972637e-03,  1.7719350811e-03,  1.8126428161e-03,
     1.8524937014e-03,  1.8914613460e-03,  1.9295197519e-03,  1.9666433305e-03,
     2.0028069200e-03,  2.0379858000e-03,  2.0721557088e-03,  2.1052928591e-03,
     2.1373739531e-03,  2.1683761987e-03,  2.1982773242e-03,  2.2270555933e-03,
     2.2546898200e-03,  2.2811593830e-03,  2.3064442399e-03,  2.3305249414e-03,
     2.3533826448e-03,  2.3749991267e-03,  2.3953567975e-03,  2.4144387128e-03,
     2.4322285864e-03,  2.4487108028e-03,  2.4638704282e-03,  2.4776932225e-03,
     2.4901656502e-03,  2.5012748911e-03,  2.5110088504e-03,  2.5193561697e-03,
     2.5263062351e-03,  2.5318491876e-03,  2.5359759318e-03,  2.5386781434e-03,
     2.5399482782e-03,  2.5397795796e-03,  2.5381660848e-03,  2.5351026330e-03,
     2.5305848708e-03,  2.5246092580e-03,  2.5171730734e-03,  2.5082744201e-03,
     2.4979122288e-03,  2.4860862633e-03,  2.4727971236e-03,  2.4580462482e-03,
     2.4418359185e-03,  2.4241692593e-03,  2.4050502412e-03,  2.3844836826e-03,
     2.3624752491e-03,  2.3390314548e-03,  2.3141596616e-03,  2.2878680788e-03,
     2.2601657612e-03,  2.2310626089e-03,  2.2005693637e-03,  2.1686976067e-03,
     2.1354597563e-03,  2.1008690627e-03,  2.0649396052e-03,  2.0276862871e-03,
     1.9891248303e-03,  1.9492717703e-03,  1.9081444495e-03,  1.8657610109e-03,
     1.8221403907e-03,  1.7773023116e-03,  1.7312672731e-03,  1.6840565449e-03,
     1.6356921569e-03,  1.5861968890e-03,  1.5355942627e-03,  1.4839085295e-03,
     1.4311646606e-03,  1.3773883350e-03,  1.3226059288e-03,  1.2668445013e-03,
     1.2101317838e-03,  1.1524961660e-03,  1.0939666815e-03,  1.0345729956e-03,
     9.7434538943e-04,  9.1331474565e-04,  8.5151253348e-04,  7.8897079241e-04,
     7.2572211666e-04,  6.6179963884e-04,  5.9723701297e-04,  5.3206839712e-04,
     4.6632843667e-04,  4.0005224544e-04,  3.3327538793e-04,  2.6603386147e-04,
     1.9836407592e-04,  1.3030283592e-04,  6.1887320653e-05, -6.8449359908e-06,
    -7.5856065182e-05, -1.4510788339e-04, -2.1456191343e-04, -2.8417940492e-04,
    -3.5392135512e-04, -4.2374853129e-04, -4.9362149112e-04, -5.6350060485e-04,
    -6.3334607783e-04, -7.0311797134e-04, -7.7277622607e-04, -8.4228068412e-04,
    -9.1159111155e-04, -9.8066722119e-04, -1.0494686958e-03, -1.1179552104e-03,
    -1.1860864557e-03, -1.2538221618e-03, -1.3211221201e-03, -1.3879462079e-03,
    -1.4542544112e-03, -1.5200068478e-03, -1.5851637911e-03, -1.6496856934e-03,
    -1.7135332088e-03, -1.7766672175e-03, -1.8390488484e-03, -1.9006395025e-03,
    -1.9614008760e-03, -2.0212949841e-03, -2.0802841828e-03, -2.1383311934e-03,
    -2.1953991240e-03, -2.2514514927e-03, -2.3064522505e-03, -2.3603658030e-03,
    -2.4131570330e-03, -2.4647913226e-03, -2.5152345751e-03, -2.5644532360e-03,
    -2.6124143155e-03, -2.6590854083e-03, -2.7044347156e-03, -2.7484310657e-03,
    -2.7910439332e-03, -2.8322434608e-03, -2.8720004779e-03, -2.9102865201e-03,
    -2.9470738489e-03, -2.9823354702e-03, -3.0160451526e-03, -3.0481774455e-03,
    -3.0787076974e-03, -3.1076120723e-03, -3.1348675672e-03, -3.1604520285e-03,
    -3.1843441676e-03, -3.2065235777e-03, -3.2269707479e-03, -3.2456670783e-03,
    -3.2625948945e-03, -3.2777374611e-03, -3.2910789951e-03, -3.3026046789e-03,
    -3.3123006729e-03, -3.3201541262e-03, -3.3261531895e-03, -3.3302870245e-03,
    -3.3325458145e-03, -3.3329207749e-03, -3.3314041612e-03, -3.3279892781e-03,
    -3.3226704878e-03, -3.3154432167e-03, -3.3063039626e-03, -3.2952503014e-03,
    -3.2822808921e-03, -3.2673954820e-03, -3.2505949120e-03, -3.2318811187e-03,
    -3.2112571393e-03, -3.1887271137e-03, -3.1642962860e-03, -3.1379710067e-03,
    -3.1097587327e-03, -3.0796680280e-03, -3.0477085626e-03, -3.0138911120e-03,
    -2.9782275545e-03, -2.9407308700e-03, -2.9014151360e-03, -2.8602955237e-03,
    -2.8173882946e-03, -2.7727107946e-03, -2.7262814488e-03, -2.6781197550e-03,
    -2.6282462771e-03, -2.5766826371e-03, -2.5234515071e-03, -2.4685766007e-03,
    -2.4120826627e-03, -2.3539954603e-03, -2.2943417712e-03, -2.2331493726e-03,
    -2.1704470300e-03, -2.1062644832e-03, -2.0406324342e-03, -1.9735825331e-03,
    -1.9051473637e-03, -1.8353604282e-03, -1.7642561325e-03, -1.6918697691e-03,
    -1.6182375007e-03, -1.5433963434e-03, -1.4673841479e-03, -1.3902395821e-03,
    -1.3120021119e-03, -1.2327119809e-03, -1.1524101917e-03, -1.0711384847e-03,
    -9.8893931695e-04, -9.0585584108e-04, -8.2193188348e-04, -7.3721192101e-04,
    -6.5174105902e-04, -5.6556500762e-04, -4.7873005738e-04, -3.9128305632e-04,
    -3.0327138418e-04, -2.1474292778e-04, -1.2574605590e-04, -3.6329592914e-05,
     5.3457207111e-05,  1.4356468668e-04,  2.3394281190e-04,  3.2454119996e-04,
     4.1530914573e-04,  5.0619565046e-04,  5.9714944958e-04,  6.8811904015e-04,
     7.7905271079e-04,  8.6989856921e-04,  9.6060457175e-04,  1.0511185526e-03,
     1.1413882532e-03,  1.2313613513e-03,  1.3209854912e-03,  1.4102083133e-03,
     1.4989774835e-03,  1.5872407247e-03,  1.6749458453e-03,  1.7620407701e-03,
     1.8484735710e-03,  1.9341924962e-03,  2.0191460011e-03,  2.1032827786e-03,
     2.1865517893e-03,  2.2689022914e-03,  2.3502838716e-03,  2.4306464741e-03,
     2.5099404314e-03,  2.5881164945e-03,  2.6651258612e-03,  2.7409202074e-03,
     2.8154517159e-03,  2.8886731051e-03,  2.9605376589e-03,  3.0309992554e-03,
     3.1000123952e-03,  3.1675322302e-03,  3.2335145917e-03,  3.2979160179e-03,
     3.3606937817e-03,  3.4218059182e-03,  3.4812112509e-03,  3.5388694192e-03,
     3.5947409036e-03,  3.6487870524e-03,  3.7009701066e-03,  3.7512532249e-03,
     3.7996005083e-03,  3.8459770245e-03,  3.8903488313e-03,  3.9326829997e-03,
     3.9729476371e-03,  4.0111119086e-03,  4.0471460593e-03,  4.0810214357e-03,
     4.1127105051e-03,  4.1421868770e-03,  4.1694253217e-03,  4.1944017893e-03,
     4.2170934279e-03,  4.2374786014e-03,  4.2555369063e-03,  4.2712491879e-03,
     4.2845975569e-03,  4.2955654030e-03,  4.3041374103e-03,  4.3102995708e-03,
     4.3140391966e-03,  4.3153449335e-03,  4.3142067716e-03,  4.3106160560e-03,
     4.3045654977e-03,  4.2960491822e-03,  4.2850625782e-03,  4.2716025458e-03,
     4.2556673434e-03,  4.2372566333e-03,  4.2163714887e-03,  4.1930143970e-03,
     4.1671892640e-03,  4.1389014179e-03,  4.1081576104e-03,  4.0749660191e-03,
     4.0393362480e-03,  4.0012793273e-03,  3.9608077121e-03,  3.9179352814e-03,
     3.8726773351e-03,  3.8250505905e-03,  3.7750731788e-03,  3.7227646387e-03,
     3.6681459120e-03,  3.6112393362e-03,  3.5520686366e-03,  3.4906589190e-03,
     3.4270366596e-03,  3.3612296957e-03,  3.2932672141e-03,  3.2231797405e-03,
     3.1509991262e-03,  3.0767585354e-03,  3.0004924311e-03,  2.9222365600e-03,
     2.8420279374e-03,  2.7599048302e-03,  2.6759067401e-03,  2.5900743856e-03,
     2.5024496834e-03,  2.4130757288e-03,  2.3219967752e-03,  2.2292582142e-03,
     2.1349065521e-03,  2.0389893895e-03,  1.9415553970e-03,  1.8426542915e-03,
     1.7423368123e-03,  1.6406546951e-03,  1.5376606469e-03,  1.4334083190e-03,
     1.3279522802e-03,  1.2213479889e-03,  1.1136517649e-03,  1.0049207603e-03,
     8.9521292975e-04,  7.8458700098e-04,  6.7310244324e-04,  5.6081943691e-04,
     4.4779884162e-04,  3.3410216337e-04,  2.1979152271e-04,  1.0492962092e-04,
    -1.0420293429e-05, -1.2619445865e-04, -2.4232863390e-04, -3.5875813517e-04,
    -4.7541786953e-04, -5.9224237117e-04, -7.0916583799e-04, -8.2612216669e-04,
    -9.4304499042e-04, -1.0598677154e-03, -1.1765235577e-03, -1.2929455815e-03,
    -1.4090667360e-03, -1.5248198936e-03, -1.6401378879e-03, -1.7549535519e-03,
    -1.8691997563e-03, -1.9828094479e-03, -2.0957156886e-03, -2.2078516933e-03,
    -2.3191508693e-03, -2.4295468546e-03, -2.5389735566e-03, -2.6473651912e-03,
    -2.7546563212e-03, -2.8607818949e-03, -2.9656772848e-03, -3.0692783263e-03,
    -3.1715213550e-03, -3.2723432462e-03, -3.3716814522e-03, -3.4694740401e-03,
    -3.5656597298e-03, -3.6601779309e-03, -3.7529687800e-03, -3.8439731778e-03,
    -3.9331328251e-03, -4.0203902595e-03, -4.1056888913e-03, -4.1889730384e-03,
    -4.2701879617e-03, -4.3492799004e-03, -4.4261961050e-03, -4.5008848723e-03,
    -4.5732955785e-03, -4.6433787111e-03, -4.7110859026e-03, -4.7763699611e-03,
    -4.8391849021e-03, -4.8994859794e-03, -4.9572297145e-03, -5.0123739266e-03,
    -5.0648777609e-03, -5.1147017175e-03, -5.1618076775e-03, -5.2061589316e-03,
    -5.2477202048e-03, -5.2864576822e-03, -5.3223390338e-03, -5.3553334381e-03,
    -5.3854116051e-03, -5.4125457987e-03, -5.4367098586e-03, -5.4578792198e-03,
    -5.4760309339e-03, -5.4911436870e-03, -5.5031978178e-03, -5.5121753357e-03,
    -5.5180599361e-03, -5.5208370164e-03, -5.5204936905e-03, -5.5170188020e-03,
    -5.5104029371e-03, -5.5006384364e-03, -5.4877194053e-03, -5.4716417236e-03,
    -5.4524030556e-03, -5.4300028556e-03, -5.4044423768e-03, -5.3757246761e-03,
    -5.3438546184e-03, -5.3088388817e-03, -5.2706859587e-03, -5.2294061591e-03,
    -5.1850116098e-03, -5.1375162552e-03, -5.0869358548e-03, -5.0332879819e-03,
    -4.9765920195e-03, -4.9168691553e-03, -4.8541423775e-03, -4.7884364667e-03,
    -4.7197779887e-03, -4.6481952865e-03, -4.5737184693e-03, -4.4963794027e-03,
    -4.4162116962e-03, -4.3332506908e-03, -4.2475334443e-03, -4.1590987172e-03,
    -4.0679869562e-03, -3.9742402772e-03, -3.8779024477e-03, -3.7790188669e-03,
    -3.6776365465e-03, -3.5738040895e-03, -3.4675716675e-03, -3.3589909987e-03,
    -3.2481153232e-03, -3.1349993787e-03, -3.0196993737e-03, -2.9022729620e-03,
    -2.7827792137e-03, -2.6612785873e-03, -2.5378329005e-03, -2.4125052986e-03,
    -2.2853602247e-03, -2.1564633864e-03, -2.0258817234e-03, -1.8936833737e-03,
    -1.7599376389e-03, -1.6247149489e-03, -1.4880868258e-03, -1.3501258470e-03,
    -1.2109056072e-03, -1.0705006807e-03, -9.2898658207e-04, -7.8643972606e-04,
    -6.4293738814e-04, -4.9855766252e-04, -3.5337942108e-04, -2.0748227107e-04,
    -6.0946512369e-05,  8.6146906166e-05,  2.3371642977e-04,  3.8167994297e-04,
     5.2995481419e-04,  6.7845794044e-04,  8.2710579401e-04,  9.7581446782e-04,
     1.1244997219e-03,  1.2730770310e-03,  1.4214616308e-03,  1.5695685660e-03,
     1.7173127381e-03,  1.8646089531e-03,  2.0113719696e-03,  2.1575165478e-03,
     2.3029574977e-03,  2.4476097276e-03,  2.5913882939e-03,  2.7342084489e-03,
     2.8759856903e-03,  3.0166358105e-03,  3.1560749451e-03,  3.2942196224e-03,
     3.4309868122e-03,  3.5662939746e-03,  3.7000591084e-03,  3.8322008010e-03,
     3.9626382751e-03,  4.0912914384e-03,  4.2180809313e-03,  4.3429281741e-03,
     4.4657554156e-03,  4.5864857793e-03,  4.7050433107e-03,  4.8213530240e-03,
     4.9353409483e-03,  5.0469341725e-03,  5.1560608919e-03,  5.2626504520e-03,
     5.3666333927e-03,  5.4679414931e-03,  5.5665078139e-03,  5.6622667398e-03,
     5.7551540226e-03,  5.8451068217e-03,  5.9320637450e-03,  6.0159648893e-03,
     6.0967518791e-03,  6.1743679056e-03,  6.2487577650e-03,  6.3198678944e-03,
     6.3876464088e-03,  6.4520431369e-03,  6.5130096546e-03,  6.5704993196e-03,
     6.6244673042e-03,  6.6748706262e-03,  6.7216681812e-03,  6.7648207715e-03,
     6.8042911355e-03,  6.8400439756e-03,  6.8720459856e-03,  6.9002658757e-03,
     6.9246743982e-03,  6.9452443710e-03,  6.9619506999e-03,  6.9747704008e-03,
     6.9836826197e-03,  6.9886686521e-03,  6.9897119613e-03,  6.9867981951e-03,
     6.9799152016e-03,  6.9690530440e-03,  6.9542040137e-03,  6.9353626424e-03,
     6.9125257136e-03,  6.8856922713e-03,  6.8548636288e-03,  6.8200433763e-03,
     6.7812373860e-03,  6.7384538169e-03,  6.6917031186e-03,  6.6409980325e-03,
     6.5863535928e-03,  6.5277871261e-03,  6.4653182491e-03,  6.3989688654e-03,
     6.3287631616e-03,  6.2547276001e-03,  6.1768909132e-03,  6.0952840940e-03,
     6.0099403861e-03,  5.9208952735e-03,  5.8281864673e-03,  5.7318538923e-03,
     5.6319396715e-03,  5.5284881103e-03,  5.4215456782e-03,  5.3111609899e-03,
     5.1973847858e-03,  5.0802699089e-03,  4.9598712836e-03,  4.8362458903e-03,
     4.7094527408e-03,  4.5795528514e-03,  4.4466092149e-03,  4.3106867717e-03,
     4.1718523799e-03,  4.0301747832e-03,  3.8857245782e-03,  3.7385741811e-03,
     3.5887977926e-03,  3.4364713610e-03,  3.2816725465e-03,  3.1244806810e-03,
     2.9649767302e-03,  2.8032432522e-03,  2.6393643563e-03,  2.4734256602e-03,
     2.3055142468e-03,  2.1357186195e-03,  1.9641286560e-03,  1.7908355633e-03,
     1.6159318285e-03,  1.4395111720e-03,  1.2616684978e-03,  1.0824998432e-03,
     9.0210232867e-04,  7.2057410575e-04,  5.3801430470e-04,  3.5452298165e-04,
     1.7020106485e-04, -1.4849700252e-05, -2.0052680570e-04, -3.8672703676e-04,
    -5.7334652905e-04, -7.6028082429e-04, -9.4742492857e-04, -1.1346733702e-03,
    -1.3219202576e-03, -1.5090593388e-03, -1.6959840603e-03, -1.8825876271e-03,
    -2.0687630619e-03, -2.2544032666e-03, -2.4394010813e-03, -2.6236493465e-03,
    -2.8070409634e-03, -2.9894689550e-03, -3.1708265279e-03, -3.3510071334e-03,
    -3.5299045285e-03, -3.7074128386e-03, -3.8834266179e-03, -4.0578409113e-03,
    -4.2305513160e-03, -4.4014540426e-03, -4.5704459759e-03, -4.7374247367e-03,
    -4.9022887419e-03, -5.0649372648e-03, -5.2252704966e-03, -5.3831896048e-03,
    -5.5385967935e-03, -5.6913953627e-03, -5.8414897665e-03, -5.9887856719e-03,
    -6.1331900164e-03, -6.2746110653e-03, -6.4129584680e-03, -6.5481433149e-03,
    -6.6800781919e-03, -6.8086772361e-03, -6.9338561893e-03, -7.0555324516e-03,
    -7.1736251339e-03, -7.2880551099e-03, -7.3987450669e-03, -7.5056195556e-03,
    -7.6086050402e-03, -7.7076299456e-03, -7.8026247058e-03, -7.8935218096e-03,
    -7.9802558456e-03, -8.0627635474e-03, -8.1409838364e-03, -8.2148578635e-03,
    -8.2843290510e-03, -8.3493431315e-03, -8.4098481869e-03, -8.4657946861e-03,
    -8.5171355209e-03, -8.5638260409e-03, -8.6058240873e-03, -8.6430900256e-03,
    -8.6755867755e-03, -8.7032798422e-03, -8.7261373428e-03, -8.7441300350e-03,
    -8.7572313415e-03, -8.7654173744e-03, -8.7686669575e-03, -8.7669616482e-03,
    -8.7602857561e-03, -8.7486263617e-03, -8.7319733335e-03, -8.7103193419e-03,
    -8.6836598739e-03, -8.6519932446e-03, -8.6153206071e-03, -8.5736459624e-03,
    -8.5269761660e-03, -8.4753209335e-03, -8.4186928447e-03, -8.3571073464e-03,
    -8.2905827524e-03, -8.2191402433e-03, -8.1428038641e-03, -8.0616005192e-03,
    -7.9755599679e-03, -7.8847148158e-03, -7.7891005065e-03, -7.6887553106e-03,
    -7.5837203134e-03, -7.4740394008e-03, -7.3597592442e-03, -7.2409292823e-03,
    -7.1176017030e-03, -6.9898314225e-03, -6.8576760631e-03, -6.7211959296e-03,
    -6.5804539840e-03, -6.4355158181e-03, -6.2864496254e-03, -6.1333261709e-03,
    -5.9762187589e-03, -5.8152032003e-03, -5.6503577777e-03, -5.4817632089e-03,
    -5.3095026091e-03, -5.1336614521e-03, -4.9543275285e-03, -4.7715909042e-03,
    -4.5855438770e-03, -4.3962809302e-03, -4.2038986879e-03, -4.0084958659e-03,
    -3.8101732225e-03, -3.6090335086e-03, -3.4051814156e-03, -3.1987235220e-03,
    -2.9897682395e-03, -2.7784257574e-03, -2.5648079850e-03, -2.3490284949e-03,
    -2.1312024630e-03, -1.9114466085e-03, -1.6898791332e-03, -1.4666196584e-03,
    -1.2417891618e-03, -1.0155099138e-03, -7.8790541116e-04, -5.5910031171e-04,
    -3.2922036707e-04, -9.8392354741e-05,  1.3325599097e-04,  3.6559604743e-04,
     5.9849837522e-04,  8.3183278796e-04,  1.0654684241e-03,  1.2992738196e-03,
     1.5331169800e-03,  1.7668654540e-03,  2.0003864075e-03,  2.2335466975e-03,
     2.4662129468e-03,  2.6982516193e-03,  2.9295290953e-03,  3.1599117466e-03,
     3.3892660138e-03,  3.6174584808e-03,  3.8443559523e-03,  4.0698255300e-03,
     4.2937346885e-03,  4.5159513525e-03,  4.7363439733e-03,  4.9547816051e-03,
     5.1711339813e-03,  5.3852715914e-03,  5.5970657561e-03,  5.8063887037e-03,
     6.0131136462e-03,  6.2171148536e-03,  6.4182677295e-03,  6.6164488854e-03,
     6.8115362148e-03,  7.0034089668e-03,  7.1919478191e-03,  7.3770349505e-03,
     7.5585541131e-03,  7.7363907031e-03,  7.9104318311e-03,  8.0805663928e-03,
     8.2466851370e-03,  8.4086807338e-03,  8.5664478428e-03,  8.7198831787e-03,
     8.8688855764e-03,  9.0133560563e-03,  9.1531978871e-03,  9.2883166480e-03,
     9.4186202905e-03,  9.5440191977e-03,  9.6644262436e-03,  9.7797568510e-03,
     9.8899290469e-03,  9.9948635184e-03,  1.0094483666e-02,  1.0188715657e-02,
     1.0277488475e-02,  1.0360733969e-02,  1.0438386904e-02,  1.0510385006e-02,
     1.0576669005e-02,  1.0637182684e-02,  1.0691872913e-02,  1.0740689698e-02,
     1.0783586211e-02,  1.0820518835e-02,  1.0851447192e-02,  1.0876334180e-02,
     1.0895146008e-02,  1.0907852218e-02,  1.0914425719e-02,  1.0914842813e-02,
     1.0909083217e-02,  1.0897130087e-02,  1.0878970039e-02,  1.0854593166e-02,
     1.0823993059e-02,  1.0787166817e-02,  1.0744115062e-02,  1.0694841953e-02,
     1.0639355189e-02,  1.0577666022e-02,  1.0509789259e-02,  1.0435743266e-02,
     1.0355549967e-02,  1.0269234846e-02,  1.0176826945e-02,  1.0078358852e-02,
     9.9738667042e-03,  9.8633901700e-03,  9.7469724424e-03,  9.6246602253e-03,
     9.4965037174e-03,  9.3625565957e-03,  9.2228759961e-03,  9.0775224920e-03,
     8.9265600708e-03,  8.7700561089e-03,  8.6080813445e-03,  8.4407098475e-03,
     8.2680189891e-03,  8.0900894082e-03,  7.9070049755e-03,  7.7188527573e-03,
     7.5257229756e-03,  7.3277089671e-03,  7.1249071407e-03,  6.9174169323e-03,
     6.7053407580e-03,  6.4887839657e-03,  6.2678547847e-03,  6.0426642729e-03,
     5.8133262643e-03,  5.5799573117e-03,  5.3426766300e-03,  5.1016060373e-03,
     4.8568698933e-03,  4.6085950376e-03,  4.3569107253e-03,  4.1019485613e-03,
     3.8438424334e-03,  3.5827284435e-03,  3.3187448370e-03,  3.0520319318e-03,
     2.7827320450e-03,  2.5109894183e-03,  2.2369501429e-03,  1.9607620818e-03,
     1.6825747918e-03,  1.4025394441e-03,  1.1208087435e-03,  8.3753684658e-04,
     5.5287927910e-04,  2.6699285204e-04, -1.9964423943e-05, -3.0783342341e-04,
    -5.9645399359e-04, -8.8566504201e-04, -1.1753046243e-03, -1.4652100342e-03,
    -1.7552178930e-03, -2.0451642403e-03, -2.3348846253e-03, -2.6242141984e-03,
    -2.9129878038e-03, -3.2010400716e-03, -3.4882055122e-03, -3.7743186083e-03,
    -4.0592139104e-03, -4.3427261298e-03, -4.6246902330e-03, -4.9049415370e-03,
    -5.1833158029e-03, -5.4596493312e-03, -5.7337790563e-03, -6.0055426411e-03,
    -6.2747785710e-03, -6.5413262490e-03, -6.8050260892e-03, -7.0657196106e-03,
    -7.3232495311e-03, -7.5774598599e-03, -7.8281959906e-03, -8.0753047934e-03,
    -8.3186347062e-03, -8.5580358260e-03, -8.7933599993e-03, -9.0244609117e-03,
    -9.2511941766e-03, -9.4734174237e-03, -9.6909903854e-03, -9.9037749839e-03,
    -1.0111635416e-02, -1.0314438239e-02, -1.0512052450e-02, -1.0704349572e-02,
    -1.0891203734e-02, -1.1072491749e-02, -1.1248093194e-02, -1.1417890487e-02,
    -1.1581768960e-02, -1.1739616940e-02, -1.1891325812e-02, -1.2036790102e-02,
    -1.2175907536e-02, -1.2308579114e-02, -1.2434709177e-02, -1.2554205468e-02,
    -1.2666979200e-02, -1.2772945112e-02, -1.2872021533e-02, -1.2964130438e-02,
    -1.3049197503e-02, -1.3127152161e-02, -1.3197927652e-02, -1.3261461073e-02,
    -1.3317693428e-02, -1.3366569670e-02, -1.3408038750e-02, -1.3442053654e-02,
    -1.3468571444e-02, -1.3487553296e-02, -1.3498964533e-02, -1.3502774662e-02,
    -1.3498957399e-02, -1.3487490700e-02, -1.3468356790e-02, -1.3441542179e-02,
    -1.3407037690e-02, -1.3364838476e-02, -1.3314944035e-02, -1.3257358223e-02,
    -1.3192089270e-02, -1.3119149787e-02, -1.3038556768e-02, -1.2950331603e-02,
    -1.2854500074e-02, -1.2751092353e-02, -1.2640143003e-02, -1.2521690971e-02,
    -1.2395779577e-02, -1.2262456508e-02, -1.2121773798e-02, -1.1973787820e-02,
    -1.1818559263e-02, -1.1656153109e-02, -1.1486638617e-02, -1.1310089291e-02,
    -1.1126582852e-02, -1.0936201212e-02, -1.0739030434e-02, -1.0535160703e-02,
    -1.0324686282e-02, -1.0107705476e-02, -9.8843205840e-03, -9.6546378600e-03,
    -9.4187674593e-03, -9.1768233914e-03, -8.9289234663e-03, -8.6751892405e-03,
    -8.4157459589e-03, -8.1507224961e-03, -7.8802512945e-03, -7.6044682999e-03,
    -7.3235128965e-03, -7.0375278378e-03, -6.7466591761e-03, -6.4510561910e-03,
    -6.1508713138e-03, -5.8462600517e-03, -5.5373809088e-03, -5.2243953063e-03,
    -4.9074674993e-03, -4.5867644929e-03, -4.2624559565e-03, -3.9347141352e-03,
    -3.6037137611e-03, -3.2696319612e-03, -2.9326481651e-03, -2.5929440101e-03,
    -2.2507032454e-03, -1.9061116343e-03, -1.5593568553e-03, -1.2106284019e-03,
    -8.6011748035e-04, -5.0801690751e-04, -1.5452100546e-04,  2.0017450330e-04,
     5.5587260220e-04,  9.1237508905e-04,  1.2694826838e-03,  1.6269951385e-03,
     1.9847113481e-03,  2.3424294613e-03,  2.6999469929e-03,  3.0570609365e-03,
     3.4135678779e-03,  3.7692641091e-03,  4.1239457432e-03,  4.4774088287e-03,
     4.8294494659e-03,  5.1798639217e-03,  5.5284487461e-03,  5.8750008881e-03,
     6.2193178124e-03,  6.5611976156e-03,  6.9004391423e-03,  7.2368421023e-03,
     7.5702071859e-03,  7.9003361801e-03,  8.2270320852e-03,  8.5500992288e-03,
     8.8693433822e-03,  9.1845718742e-03,  9.4955937054e-03,  9.8022196614e-03,
     1.0104262426e-02,  1.0401536693e-02,  1.0693859278e-02,  1.0981049227e-02,
     1.1262927929e-02,  1.1539319221e-02,  1.1810049499e-02,  1.2074947822e-02,
     1.2333846017e-02,  1.2586578784e-02,  1.2832983798e-02,  1.3072901813e-02,
     1.3306176754e-02,  1.3532655827e-02,  1.3752189604e-02,  1.3964632128e-02,
     1.4169840997e-02,  1.4367677467e-02,  1.4558006531e-02,  1.4740697013e-02,
     1.4915621654e-02,  1.5082657196e-02,  1.5241684462e-02,  1.5392588439e-02,
     1.5535258355e-02,  1.5669587755e-02,  1.5795474573e-02,  1.5912821209e-02,
     1.6021534590e-02,  1.6121526244e-02,  1.6212712361e-02,  1.6295013856e-02,
     1.6368356426e-02,  1.6432670611e-02,  1.6487891845e-02,  1.6533960510e-02,
     1.6570821984e-02,  1.6598426688e-02,  1.6616730129e-02,  1.6625692942e-02,
     1.6625280926e-02,  1.6615465084e-02,  1.6596221649e-02,  1.6567532121e-02,
     1.6529383288e-02,  1.6481767251e-02,  1.6424681450e-02,  1.6358128673e-02,
     1.6282117080e-02,  1.6196660208e-02,  1.6101776987e-02,  1.5997491737e-02,
     1.5883834181e-02,  1.5760839435e-02,  1.5628548014e-02,  1.5487005820e-02,
     1.5336264132e-02,  1.5176379600e-02,  1.5007414222e-02,  1.4829435331e-02,
     1.4642515570e-02,  1.4446732869e-02,  1.4242170417e-02,  1.4028916631e-02,
     1.3807065121e-02,  1.3576714657e-02,  1.3337969123e-02,  1.3090937479e-02,
     1.2835733713e-02,  1.2572476793e-02,  1.2301290612e-02,  1.2022303940e-02,
     1.1735650356e-02,  1.1441468197e-02,  1.1139900488e-02,  1.0831094881e-02,
     1.0515203578e-02,  1.0192383266e-02,  9.8627950398e-03,  9.5266043239e-03,
     9.1839807920e-03,  8.8350982852e-03,  8.4801347257e-03,  8.1192720288e-03,
     7.7526960125e-03,  7.3805963043e-03,  7.0031662453e-03,  6.6206027929e-03,
     6.2331064197e-03,  5.8408810113e-03,  5.4441337616e-03,  5.0430750651e-03,
     4.6379184087e-03,  4.2288802593e-03,  3.8161799513e-03,  3.4000395709e-03,
     2.9806838390e-03,  2.5583399921e-03,  2.1332376614e-03,  1.7056087507e-03,
     1.2756873114e-03,  8.4370941704e-04,  4.0991303625e-04, -2.5462096983e-05,
    -4.6217461238e-04, -8.9998163592e-04, -1.3386389221e-03, -1.7779009881e-03,
    -2.2175212489e-03, -2.6572521536e-03, -3.0968453219e-03, -3.5360516830e-03,
    -3.9746216138e-03, -4.4123050786e-03, -4.8488517693e-03, -5.2840112465e-03,
    -5.7175330804e-03, -6.1491669928e-03, -6.5786629990e-03, -7.0057715502e-03,
    -7.4302436765e-03, -7.8518311288e-03, -8.2702865221e-03, -8.6853634778e-03,
    -9.0968167665e-03, -9.5044024497e-03, -9.9078780229e-03, -1.0307002556e-02,
    -1.0701536835e-02, -1.1091243504e-02, -1.1475887201e-02, -1.1855234704e-02,
    -1.2229055062e-02, -1.2597119738e-02, -1.2959202744e-02, -1.3315080777e-02,
    -1.3664533352e-02, -1.4007342939e-02, -1.4343295091e-02, -1.4672178578e-02,
    -1.4993785513e-02, -1.5307911484e-02, -1.5614355680e-02, -1.5912921009e-02,
    -1.6203414232e-02, -1.6485646073e-02, -1.6759431346e-02, -1.7024589071e-02,
    -1.7280942586e-02, -1.7528319663e-02, -1.7766552619e-02, -1.7995478423e-02,
    -1.8214938805e-02, -1.8424780359e-02, -1.8624854645e-02, -1.8815018286e-02,
    -1.8995133070e-02, -1.9165066037e-02, -1.9324689578e-02, -1.9473881516e-02,
    -1.9612525199e-02, -1.9740509577e-02, -1.9857729287e-02, -1.9964084725e-02,
    -2.0059482125e-02, -2.0143833627e-02, -2.0217057346e-02, -2.0279077436e-02,
    -2.0329824151e-02, -2.0369233906e-02, -2.0397249329e-02, -2.0413819313e-02,
    -2.0418899065e-02, -2.0412450152e-02, -2.0394440539e-02, -2.0364844629e-02,
    -2.0323643298e-02, -2.0270823923e-02, -2.0206380412e-02, -2.0130313225e-02,
    -2.0042629398e-02, -1.9943342553e-02, -1.9832472918e-02, -1.9710047327e-02,
    -1.9576099236e-02, -1.9430668713e-02, -1.9273802446e-02, -1.9105553729e-02,
    -1.8925982457e-02, -1.8735155111e-02, -1.8533144741e-02, -1.8320030942e-02,
    -1.8095899836e-02, -1.7860844035e-02, -1.7614962616e-02, -1.7358361079e-02,
    -1.7091151312e-02, -1.6813451543e-02, -1.6525386298e-02, -1.6227086344e-02,
    -1.5918688638e-02, -1.5600336268e-02, -1.5272178392e-02, -1.4934370169e-02,
    -1.4587072693e-02, -1.4230452921e-02, -1.3864683591e-02, -1.3489943149e-02,
    -1.3106415663e-02, -1.2714290734e-02, -1.2313763409e-02, -1.1905034088e-02,
    -1.1488308423e-02, -1.1063797223e-02, -1.0631716346e-02, -1.0192286597e-02,
    -9.7457336164e-03, -9.2922877663e-03, -8.8321840164e-03, -8.3656618254e-03,
    -7.8929650188e-03, -7.4143416648e-03, -6.9300439476e-03, -6.4403280370e-03,
    -5.9454539561e-03, -5.4456854470e-03, -4.9412898318e-03, -4.4325378739e-03,
    -3.9197036349e-03, -3.4030643303e-03, -2.8829001820e-03, -2.3594942703e-03,
    -1.8331323817e-03, -1.3041028564e-03, -7.7269643332e-04, -2.3920609248e-04,
     2.9607310255e-04,  8.3284416544e-04,  1.3708083491e-03,  1.9096653087e-03,
     2.4491132664e-03,  2.9888491777e-03,  3.5285688986e-03,  4.0679673534e-03,
     4.6067387054e-03,  5.1445765260e-03,  5.6811739666e-03,  6.2162239309e-03,
     6.7494192468e-03,  7.2804528401e-03,  7.8090179086e-03,  8.3348080951e-03,
     8.8575176630e-03,  9.3768416703e-03,  9.8924761442e-03,  1.0404118256e-02,
     1.0911466495e-02,  1.1414220844e-02,  1.1912082952e-02,  1.2404756310e-02,
     1.2891946419e-02,  1.3373360971e-02,  1.3848710013e-02,  1.4317706119e-02,
     1.4780064566e-02,  1.5235503497e-02,  1.5683744090e-02,  1.6124510728e-02,
     1.6557531161e-02,  1.6982536671e-02,  1.7399262236e-02,  1.7807446689e-02,
     1.8206832880e-02,  1.8597167830e-02,  1.8978202889e-02,  1.9349693891e-02,
     1.9711401303e-02,  2.0063090373e-02,  2.0404531284e-02,  2.0735499292e-02,
     2.1055774872e-02,  2.1365143858e-02,  2.1663397578e-02,  2.1950332993e-02,
     2.2225752823e-02,  2.2489465683e-02,  2.2741286203e-02,  2.2981035157e-02,
     2.3208539577e-02,  2.3423632875e-02,  2.3626154955e-02,  2.3815952322e-02,
     2.3992878190e-02,  2.4156792588e-02,  2.4307562455e-02,  2.4445061742e-02,
     2.4569171500e-02,  2.4679779971e-02,  2.4776782676e-02,  2.4860082491e-02,
     2.4929589729e-02,  2.4985222211e-02,  2.5026905338e-02,  2.5054572154e-02,
     2.5068163409e-02,  2.5067627614e-02,  2.5052921100e-02,  2.5024008058e-02,
     2.4980860591e-02,  2.4923458750e-02,  2.4851790573e-02,  2.4765852113e-02,
     2.4665647467e-02,  2.4551188799e-02,  2.4422496357e-02,  2.4279598487e-02,
     2.4122531641e-02,  2.3951340385e-02,  2.3766077394e-02,  2.3566803452e-02,
     2.3353587440e-02,  2.3126506322e-02,  2.2885645130e-02,  2.2631096936e-02,
     2.2362962830e-02,  2.2081351881e-02,  2.1786381109e-02,  2.1478175434e-02,
     2.1156867639e-02,  2.0822598311e-02,  2.0475515794e-02,  2.0115776123e-02,
     1.9743542964e-02,  1.9358987543e-02,  1.8962288574e-02,  1.8553632181e-02,
     1.8133211815e-02,  1.7701228171e-02,  1.7257889093e-02,  1.6803409483e-02,
     1.6338011198e-02,  1.5861922949e-02,  1.5375380194e-02,  1.4878625023e-02,
     1.4371906043e-02,  1.3855478263e-02,  1.3329602961e-02,  1.2794547566e-02,
     1.2250585517e-02,  1.1697996137e-02,  1.1137064485e-02,  1.0568081219e-02,
     9.9913424465e-03,  9.4071495767e-03,  8.8158091654e-03,  8.2176327591e-03,
     7.6129367342e-03,  7.0020421346e-03,  6.3852745044e-03,  5.7629637189e-03,
     5.1354438119e-03,  4.5030528007e-03,  3.8661325076e-03,  3.2250283792e-03,
     2.5800893034e-03,  1.9316674227e-03,  1.2801179467e-03,  6.2579896088e-04,
    -3.0928766166e-05, -6.8970197709e-04, -1.3501551235e-03, -2.0119205646e-03,
    -2.6746287673e-03, -3.3379085095e-03, -4.0013870832e-03, -4.6646904999e-03,
    -5.3274436982e-03, -5.9892707506e-03, -6.6497950734e-03, -7.3086396367e-03,
    -7.9654271752e-03, -8.6197804001e-03, -9.2713222117e-03, -9.9196759124e-03,
    -1.0564465420e-02, -1.1205315482e-02, -1.1841851888e-02, -1.2473701688e-02,
    -1.3100493399e-02, -1.3721857227e-02, -1.4337425274e-02, -1.4946831758e-02,
    -1.5549713216e-02, -1.6145708727e-02, -1.6734460114e-02, -1.7315612161e-02,
    -1.7888812818e-02, -1.8453713409e-02, -1.9009968845e-02, -1.9557237822e-02,
    -2.0095183029e-02, -2.0623471349e-02, -2.1141774063e-02, -2.1649767044e-02,
    -2.2147130956e-02, -2.2633551452e-02, -2.3108719360e-02, -2.3572330881e-02,
    -2.4024087770e-02, -2.4463697527e-02, -2.4890873573e-02, -2.5305335439e-02,
    -2.5706808936e-02, -2.6095026330e-02, -2.6469726516e-02, -2.6830655183e-02,
    -2.7177564980e-02, -2.7510215678e-02, -2.7828374324e-02, -2.8131815399e-02,
    -2.8420320966e-02, -2.8693680818e-02, -2.8951692618e-02, -2.9194162041e-02,
    -2.9420902908e-02, -2.9631737314e-02, -2.9826495756e-02, -3.0005017255e-02,
    -3.0167149472e-02, -3.0312748824e-02, -3.0441680589e-02, -3.0553819011e-02,
    -3.0649047400e-02, -3.0727258226e-02, -3.0788353206e-02, -3.0832243391e-02,
    -3.0858849247e-02, -3.0868100725e-02, -3.0859937334e-02, -3.0834308202e-02,
    -3.0791172140e-02, -3.0730497690e-02, -3.0652263177e-02, -3.0556456753e-02,
    -3.0443076431e-02, -3.0312130120e-02, -3.0163635652e-02, -2.9997620801e-02,
    -2.9814123301e-02, -2.9613190858e-02, -2.9394881148e-02, -2.9159261826e-02,
    -2.8906410509e-02, -2.8636414773e-02, -2.8349372130e-02, -2.8045390005e-02,
    -2.7724585710e-02, -2.7387086406e-02, -2.7033029066e-02, -2.6662560426e-02,
    -2.6275836935e-02, -2.5873024697e-02, -2.5454299411e-02, -2.5019846298e-02,
    -2.4569860033e-02, -2.4104544660e-02, -2.3624113510e-02, -2.3128789109e-02,
    -2.2618803085e-02, -2.2094396065e-02, -2.1555817566e-02, -2.1003325888e-02,
    -2.0437187991e-02, -1.9857679379e-02, -1.9265083966e-02, -1.8659693949e-02,
    -1.8041809664e-02, -1.7411739449e-02, -1.6769799492e-02, -1.6116313682e-02,
    -1.5451613447e-02, -1.4776037597e-02, -1.4089932153e-02, -1.3393650176e-02,
    -1.2687551594e-02, -1.1972003019e-02, -1.1247377564e-02, -1.0514054651e-02,
    -9.7724198238e-03, -9.0228645441e-03, -8.2657859957e-03, -7.5015868771e-03,
    -6.7306751936e-03, -5.9534640450e-03, -5.1703714102e-03, -4.3818199271e-03,
    -3.5882366713e-03, -2.7900529290e-03, -1.9877039697e-03, -1.1816288136e-03,
    -3.7226999674e-04,  4.3992666514e-04,  1.2545123175e-03,  2.0710353092e-03,
     2.8890414367e-03,  3.7080741913e-03,  4.5276750085e-03,  5.3473835179e-03,
     6.1667377968e-03,  6.9852746248e-03,  7.8025297386e-03,  8.6180380906e-03,
     9.4313341068e-03,  1.0241951947e-02,  1.1049425764e-02,  1.1853289969e-02,
     1.2653079488e-02,  1.3448330030e-02,  1.4238578348e-02,  1.5023362502e-02,
     1.5802222123e-02,  1.6574698675e-02,  1.7340335724e-02,  1.8098679193e-02,
     1.8849277631e-02,  1.9591682470e-02,  2.0325448291e-02,  2.1050133080e-02,
     2.1765298490e-02,  2.2470510096e-02,  2.3165337655e-02,  2.3849355360e-02,
     2.4522142090e-02,  2.5183281667e-02,  2.5832363101e-02,  2.6468980839e-02,
     2.7092735013e-02,  2.7703231679e-02,  2.8300083057e-02,  2.8882907774e-02,
     2.9451331092e-02,  3.0004985145e-02,  3.0543509166e-02,  3.1066549709e-02,
     3.1573760879e-02,  3.2064804544e-02,  3.2539350553e-02,  3.2997076944e-02,
     3.3437670157e-02,  3.3860825233e-02,  3.4266246013e-02,  3.4653645336e-02,
     3.5022745225e-02,  3.5373277079e-02,  3.5704981849e-02,  3.6017610214e-02,
     3.6310922761e-02,  3.6584690139e-02,  3.6838693233e-02,  3.7072723311e-02,
     3.7286582181e-02,  3.7480082334e-02,  3.7653047085e-02,  3.7805310707e-02,
     3.7936718561e-02,  3.8047127218e-02,  3.8136404576e-02,  3.8204429974e-02,
     3.8251094294e-02,  3.8276300063e-02,  3.8279961542e-02,  3.8262004817e-02,
     3.8222367879e-02,  3.8161000693e-02,  3.8077865273e-02,  3.7972935738e-02,
     3.7846198367e-02,  3.7697651648e-02,  3.7527306321e-02,  3.7335185406e-02,
     3.7121324241e-02,  3.6885770491e-02,  3.6628584174e-02,  3.6349837657e-02,
     3.6049615667e-02,  3.5728015275e-02,  3.5385145891e-02,  3.5021129237e-02,
     3.4636099322e-02,  3.4230202412e-02,  3.3803596980e-02,  3.3356453666e-02,
     3.2888955218e-02,  3.2401296431e-02,  3.1893684075e-02,  3.1366336826e-02,
     3.0819485176e-02,  3.0253371348e-02,  2.9668249197e-02,  2.9064384107e-02,
     2.8442052883e-02,  2.7801543632e-02,  2.7143155639e-02,  2.6467199240e-02,
     2.5773995681e-02,  2.5063876979e-02,  2.4337185769e-02,  2.3594275150e-02,
     2.2835508519e-02,  2.2061259407e-02,  2.1271911301e-02,  2.0467857462e-02,
     1.9649500740e-02,  1.8817253381e-02,  1.7971536824e-02,  1.7112781501e-02,
     1.6241426625e-02,  1.5357919970e-02,  1.4462717656e-02,  1.3556283917e-02,
     1.2639090868e-02,  1.1711618274e-02,  1.0774353299e-02,  9.8277902647e-03,
     8.8724303956e-03,  7.9087815622e-03,  6.9373580202e-03,  5.9586801435e-03,
     4.9732741549e-03,  3.9816718507e-03,  2.9844103231e-03,  1.9820316773e-03,
     9.7508274565e-04, -3.5885202143e-05, -1.0503167520e-03, -2.0676526422e-03,
    -3.0873300626e-03, -4.1087829572e-03, -5.1314423297e-03, -6.1547365516e-03,
    -7.1780916719e-03, -8.2009317312e-03, -9.2226790761e-03, -1.0242754676e-02,
    -1.1260578442e-02, -1.2275569548e-02, -1.3287146753e-02, -1.4294728722e-02,
    -1.5297734354e-02, -1.6295583104e-02, -1.7287695314e-02, -1.8273492534e-02,
    -1.9252397852e-02, -2.0223836223e-02, -2.1187234795e-02, -2.2142023236e-02,
    -2.3087634062e-02, -2.4023502965e-02, -2.4949069134e-02, -2.5863775589e-02,
    -2.6767069495e-02, -2.7658402493e-02, -2.8537231019e-02, -2.9403016621e-02,
    -3.0255226281e-02, -3.1093332733e-02, -3.1916814770e-02, -3.2725157562e-02,
    -3.3517852966e-02, -3.4294399828e-02, -3.5054304290e-02, -3.5797080089e-02,
    -3.6522248860e-02, -3.7229340422e-02, -3.7917893077e-02, -3.8587453890e-02,
    -3.9237578977e-02, -3.9867833783e-02, -4.0477793356e-02, -4.1067042618e-02,
    -4.1635176632e-02, -4.2181800862e-02, -4.2706531427e-02, -4.3208995359e-02,
    -4.3688830843e-02, -4.4145687460e-02, -4.4579226422e-02, -4.4989120803e-02,
    -4.5375055760e-02, -4.5736728754e-02, -4.6073849760e-02, -4.6386141473e-02,
    -4.6673339508e-02, -4.6935192590e-02, -4.7171462745e-02, -4.7381925474e-02,
    -4.7566369932e-02, -4.7724599086e-02, -4.7856429880e-02, -4.7961693384e-02,
    -4.8040234936e-02, -4.8091914281e-02, -4.8116605701e-02, -4.8114198134e-02,
    -4.8084595288e-02, -4.8027715749e-02, -4.7943493077e-02, -4.7831875898e-02,
    -4.7692827984e-02, -4.7526328328e-02, -4.7332371210e-02, -4.7110966257e-02,
    -4.6862138486e-02, -4.6585928350e-02, -4.6282391770e-02, -4.5951600158e-02,
    -4.5593640432e-02, -4.5208615024e-02, -4.4796641879e-02, -4.4357854443e-02,
    -4.3892401646e-02, -4.3400447874e-02, -4.2882172933e-02, -4.2337772002e-02,
    -4.1767455586e-02, -4.1171449444e-02, -4.0549994530e-02, -3.9903346903e-02,
    -3.9231777646e-02, -3.8535572764e-02, -3.7815033086e-02, -3.7070474142e-02,
    -3.6302226050e-02, -3.5510633376e-02, -3.4696055003e-02, -3.3858863977e-02,
    -3.2999447355e-02, -3.2118206037e-02, -3.1215554598e-02, -3.0291921101e-02,
    -2.9347746916e-02, -2.8383486514e-02, -2.7399607270e-02, -2.6396589247e-02,
    -2.5374924973e-02, -2.4335119217e-02, -2.3277688751e-02, -2.2203162108e-02,
    -2.1112079330e-02, -2.0004991711e-02, -1.8882461530e-02, -1.7745061784e-02,
    -1.6593375902e-02, -1.5427997466e-02, -1.4249529914e-02, -1.3058586243e-02,
    -1.1855788704e-02, -1.0641768490e-02, -9.4171654199e-03, -8.1826276123e-03,
    -6.9388111592e-03, -5.6863797901e-03, -4.4260045317e-03, -3.1583633634e-03,
    -1.8841408659e-03, -6.0402786624e-04,  6.8127892215e-04,  1.9710772644e-03,
     3.2646597732e-03,  4.5613142820e-03,  5.8603242218e-03,  7.1609690030e-03,
     8.4625243991e-03,  9.7642629355e-03,  1.1065454280e-02,  1.2365365639e-02,
     1.3663262150e-02,  1.4958407286e-02,  1.6250063257e-02,  1.7537491410e-02,
     1.8819952640e-02,  2.0096707794e-02,  2.1367018085e-02,  2.2630145498e-02,
     2.3885353203e-02,  2.5131905971e-02,  2.6369070581e-02,  2.7596116242e-02,
     2.8812314997e-02,  3.0016942147e-02,  3.1209276656e-02,  3.2388601569e-02,
     3.3554204425e-02,  3.4705377664e-02,  3.5841419041e-02,  3.6961632035e-02,
     3.8065326252e-02,  3.9151817837e-02,  4.0220429871e-02,  4.1270492778e-02,
     4.2301344717e-02,  4.3312331982e-02,  4.4302809394e-02,  4.5272140691e-02,
     4.6219698913e-02,  4.7144866782e-02,  4.8047037086e-02,  4.8925613051e-02,
     4.9780008707e-02,  5.0609649261e-02,  5.1413971450e-02,  5.2192423905e-02,
     5.2944467495e-02,  5.3669575673e-02,  5.4367234819e-02,  5.5036944572e-02,
     5.5678218157e-02,  5.6290582708e-02,  5.6873579580e-02,  5.7426764663e-02,
     5.7949708679e-02,  5.8441997479e-02,  5.8903232330e-02,  5.9333030196e-02,
     5.9731024008e-02,  6.0096862935e-02,  6.0430212634e-02,  6.0730755502e-02,
     6.0998190920e-02,  6.1232235479e-02,  6.1432623212e-02,  6.1599105801e-02,
     6.1731452789e-02,  6.1829451776e-02,  6.1892908605e-02,  6.1921647547e-02,
     6.1915511462e-02,  6.1874361967e-02,  6.1798079584e-02,  6.1686563877e-02,
     6.1539733590e-02,  6.1357526761e-02,  6.1139900837e-02,  6.0886832773e-02,
     6.0598319124e-02,  6.0274376124e-02,  5.9915039753e-02,  5.9520365803e-02,
     5.9090429917e-02,  5.8625327636e-02,  5.8125174421e-02,  5.7590105670e-02,
     5.7020276723e-02,  5.6415862861e-02,  5.5777059284e-02,  5.5104081086e-02,
     5.4397163219e-02,  5.3656560441e-02,  5.2882547258e-02,  5.2075417853e-02,
     5.1235486000e-02,  5.0363084978e-02,  4.9458567459e-02,  4.8522305400e-02,
     4.7554689914e-02,  4.6556131130e-02,  4.5527058052e-02,  4.4467918394e-02,
     4.3379178415e-02,  4.2261322736e-02,  4.1114854149e-02,  3.9940293419e-02,
     3.8738179066e-02,  3.7509067149e-02,  3.6253531027e-02,  3.4972161119e-02,
     3.3665564650e-02,  3.2334365386e-02,  3.0979203359e-02,  2.9600734589e-02,
     2.8199630784e-02,  2.6776579038e-02,  2.5332281521e-02,  2.3867455156e-02,
     2.2382831284e-02,  2.0879155326e-02,  1.9357186434e-02,  1.7817697131e-02,
     1.6261472942e-02,  1.4689312023e-02,  1.3102024771e-02,  1.1500433436e-02,
     9.8853717179e-03,  8.2576843598e-03,  6.6182267312e-03,  4.9678644051e-03,
     3.3074727272e-03,  1.6379363784e-03, -3.9851069993e-05, -1.7249876071e-03,
    -3.4165632398e-03, -5.1136604573e-03, -6.8153546992e-03, -8.5207148330e-03,
    -1.0228803633e-02, -1.1938678268e-02, -1.3649390792e-02, -1.5359988639e-02,
    -1.7069515125e-02, -1.8777009954e-02, -2.0481509723e-02, -2.2182048438e-02,
    -2.3877658030e-02, -2.5567368871e-02, -2.7250210301e-02, -2.8925211150e-02,
    -3.0591400266e-02, -3.2247807047e-02, -3.3893461969e-02, -3.5527397123e-02,
    -3.7148646747e-02, -3.8756247763e-02, -4.0349240312e-02, -4.1926668294e-02,
    -4.3487579902e-02, -4.5031028162e-02, -4.6556071467e-02, -4.8061774116e-02,
    -4.9547206846e-02, -5.1011447372e-02, -5.2453580912e-02, -5.3872700723e-02,
    -5.5267908628e-02, -5.6638315543e-02, -5.7983042000e-02, -5.9301218669e-02,
    -6.0591986872e-02, -6.1854499102e-02, -6.3087919531e-02, -6.4291424515e-02,
    -6.5464203097e-02, -6.6605457502e-02, -6.7714403634e-02, -6.8790271558e-02,
    -6.9832305984e-02, -7.0839766740e-02, -7.1811929246e-02, -7.2748084971e-02,
    -7.3647541898e-02, -7.4509624964e-02, -7.5333676511e-02, -7.6119056718e-02,
    -7.6865144025e-02, -7.7571335560e-02, -7.8237047544e-02, -7.8861715699e-02,
    -7.9444795637e-02, -7.9985763251e-02, -8.0484115088e-02, -8.0939368718e-02,
    -8.1351063088e-02, -8.1718758875e-02, -8.2042038817e-02, -8.2320508047e-02,
    -8.2553794402e-02, -8.2741548737e-02, -8.2883445214e-02, -8.2979181588e-02,
    -8.3028479480e-02, -8.3031084639e-02, -8.2986767190e-02, -8.2895321870e-02,
    -8.2756568258e-02, -8.2570350984e-02, -8.2336539933e-02, -8.2055030432e-02,
    -8.1725743426e-02, -8.1348625641e-02, -8.0923649735e-02, -8.0450814434e-02,
    -7.9930144654e-02, -7.9361691617e-02, -7.8745532941e-02, -7.8081772729e-02,
    -7.7370541634e-02, -7.6611996920e-02, -7.5806322501e-02, -7.4953728970e-02,
    -7.4054453614e-02, -7.3108760412e-02, -7.2116940024e-02, -7.1079309764e-02,
    -6.9996213552e-02, -6.8868021863e-02, -6.7695131656e-02, -6.6477966285e-02,
    -6.5216975406e-02, -6.3912634857e-02, -6.2565446533e-02, -6.1175938245e-02,
    -5.9744663563e-02, -5.8272201642e-02, -5.6759157041e-02, -5.5206159523e-02,
    -5.3613863840e-02, -5.1982949506e-02, -5.0314120558e-02, -4.8608105295e-02,
    -4.6865656014e-02, -4.5087548726e-02, -4.3274582856e-02, -4.1427580933e-02,
    -3.9547388269e-02, -3.7634872617e-02, -3.5690923824e-02, -3.3716453465e-02,
    -3.1712394465e-02, -2.9679700709e-02, -2.7619346642e-02, -2.5532326848e-02,
    -2.3419655629e-02, -2.1282366558e-02, -1.9121512031e-02, -1.6938162802e-02,
    -1.4733407503e-02, -1.2508352163e-02, -1.0264119704e-02, -8.0018494307e-03,
    -5.7226965139e-03, -3.4278314541e-03, -1.1184395412e-03,  1.2042796983e-03,
     3.5391130636e-03,  5.8848342528e-03,  8.2402044434e-03,  1.0603972885e-02,
     1.2974877497e-02,  1.5351645482e-02,  1.7732993936e-02,  2.0117630481e-02,
     2.2504253895e-02,  2.4891554752e-02,  2.7278216073e-02,  2.9662913978e-02,
     3.2044318353e-02,  3.4421093512e-02,  3.6791898877e-02,  3.9155389655e-02,
     4.1510217527e-02,  4.3855031333e-02,  4.6188477777e-02,  4.8509202116e-02,
     5.0815848874e-02,  5.3107062544e-02,  5.5381488303e-02,  5.7637772724e-02,
     5.9874564497e-02,  6.2090515144e-02,  6.4284279745e-02,  6.6454517662e-02,
     6.8599893258e-02,  7.0719076632e-02,  7.2810744337e-02,  7.4873580114e-02,
     7.6906275613e-02,  7.8907531125e-02,  8.0876056301e-02,  8.2810570882e-02,
     8.4709805420e-02,  8.6572501997e-02,  8.8397414947e-02,  9.0183311569e-02,
     9.1928972841e-02,  9.3633194128e-02,  9.5294785892e-02,  9.6912574388e-02,
     9.8485402366e-02,  1.0001212976e-01,  1.0149163438e-01,  1.0292281257e-01,
     1.0430457994e-01,  1.0563587196e-01,  1.0691564468e-01,  1.0814287536e-01,
     1.0931656313e-01,  1.1043572962e-01,  1.1149941959e-01,  1.1250670158e-01,
     1.1345666847e-01,  1.1434843815e-01,  1.1518115408e-01,  1.1595398588e-01,
     1.1666612991e-01,  1.1731680984e-01,  1.1790527720e-01,  1.1843081192e-01,
     1.1889272288e-01,  1.1929034841e-01,  1.1962305681e-01,  1.1989024684e-01,
     1.2009134822e-01,  1.2022582209e-01,  1.2029316146e-01,  1.2029289166e-01,
     1.2022457079e-01,  1.2008779012e-01,  1.1988217446e-01,  1.1960738263e-01,
     1.1926310776e-01,  1.1884907765e-01,  1.1836505519e-01,  1.1781083859e-01,
     1.1718626175e-01,  1.1649119454e-01,  1.1572554309e-01,  1.1488925002e-01,
     1.1398229475e-01,  1.1300469367e-01,  1.1195650036e-01,  1.1083780584e-01,
     1.0964873868e-01,  1.0838946520e-01,  1.0706018962e-01,  1.0566115416e-01,
     1.0419263918e-01,  1.0265496323e-01,  1.0104848316e-01,  9.9373594172e-02,
     9.7630729822e-02,  9.5820362082e-02,  9.3943001311e-02,  9.1999196249e-02,
     8.9989533974e-02,  8.7914639851e-02,  8.5775177452e-02,  8.3571848469e-02,
     8.1305392597e-02,  7.8976587406e-02,  7.6586248191e-02,  7.4135227804e-02,
     7.1624416469e-02,  6.9054741574e-02,  6.6427167446e-02,  6.3742695112e-02,
     6.1002362035e-02,  5.8207241835e-02,  5.5358443989e-02,  5.2457113515e-02,
     4.9504430638e-02,  4.6501610434e-02,  4.3449902460e-02,  4.0350590364e-02,
     3.7204991479e-02,  3.4014456392e-02,  3.0780368507e-02,  2.7504143583e-02,
     2.4187229253e-02,  2.0831104530e-02,  1.7437279298e-02,  1.4007293779e-02,
     1.0542717988e-02,  7.0451511706e-03,  3.5162212259e-03, -4.2415889919e-05,
    -3.6290767732e-03, -7.2420511994e-03, -1.0879602764e-02, -1.4539969541e-02,
    -1.8221364752e-02, -2.1921977453e-02, -2.5639973237e-02, -2.9373494945e-02,
    -3.3120663398e-02, -3.6879578136e-02, -4.0648318176e-02, -4.4424942777e-02,
    -4.8207492221e-02, -5.1993988609e-02, -5.5782436662e-02, -5.9570824539e-02,
    -6.3357124664e-02, -6.7139294567e-02, -7.0915277727e-02, -7.4683004439e-02,
    -7.8440392679e-02, -8.2185348981e-02, -8.5915769328e-02, -8.9629540047e-02,
    -9.3324538715e-02, -9.6998635069e-02, -1.0064969193e-01, -1.0427556612e-01,
    -1.0787410942e-01, -1.1144316948e-01, -1.1498059078e-01, -1.1848421560e-01,
    -1.2195188493e-01, -1.2538143948e-01, -1.2877072063e-01, -1.3211757138e-01,
    -1.3541983734e-01, -1.3867536772e-01, -1.4188201627e-01, -1.4503764229e-01,
    -1.4814011161e-01, -1.5118729757e-01, -1.5417708199e-01, -1.5710735618e-01,
    -1.5997602188e-01, -1.6278099229e-01, -1.6552019305e-01, -1.6819156318e-01,
    -1.7079305608e-01, -1.7332264052e-01, -1.7577830160e-01, -1.7815804171e-01,
    -1.8045988153e-01, -1.8268186093e-01, -1.8482204000e-01, -1.8687849993e-01,
    -1.8884934403e-01, -1.9073269860e-01, -1.9252671390e-01, -1.9422956508e-01,
    -1.9583945308e-01, -1.9735460555e-01, -1.9877327775e-01, -2.0009375344e-01,
    -2.0131434579e-01, -2.0243339822e-01, -2.0344928527e-01, -2.0436041348e-01,
    -2.0516522220e-01, -2.0586218446e-01, -2.0644980775e-01, -2.0692663486e-01,
    -2.0729124467e-01, -2.0754225290e-01, -2.0767831293e-01, -2.0769811651e-01,
    -2.0760039456e-01, -2.0738391783e-01, -2.0704749764e-01, -2.0658998660e-01,
    -2.0601027926e-01, -2.0530731281e-01, -2.0448006766e-01, -2.0352756818e-01,
    -2.0244888321e-01, -2.0124312672e-01, -1.9990945840e-01, -1.9844708416e-01,
    -1.9685525675e-01, -1.9513327624e-01, -1.9328049053e-01, -1.9129629587e-01,
    -1.8918013729e-01, -1.8693150908e-01, -1.8454995521e-01, -1.8203506974e-01,
    -1.7938649720e-01, -1.7660393299e-01, -1.7368712370e-01, -1.7063586742e-01,
    -1.6745001412e-01, -1.6412946585e-01, -1.6067417705e-01, -1.5708415480e-01,
    -1.5335945901e-01, -1.4950020263e-01, -1.4550655183e-01, -1.4137872616e-01,
    -1.3711699866e-01, -1.3272169599e-01, -1.2819319851e-01, -1.2353194032e-01,
    -1.1873840935e-01, -1.1381314734e-01, -1.0875674982e-01, -1.0356986614e-01,
    -9.8253199367e-02, -9.2807506219e-02, -8.7233596988e-02, -8.1532335403e-02,
    -7.5704638489e-02, -6.9751476402e-02, -6.3673872237e-02, -5.7472901815e-02,
    -5.1149693447e-02, -4.4705427678e-02, -3.8141337000e-02, -3.1458705549e-02,
    -2.4658868780e-02, -1.7743213112e-02, -1.0713175559e-02, -3.5702433330e-03,
     3.6840465721e-03,  1.1048107822e-02,  1.8520305205e-02,  2.6098955109e-02,
     3.3782326036e-02,  4.1568639124e-02,  4.9456068692e-02,  5.7442742818e-02,
     6.5526743922e-02,  7.3706109379e-02,  8.1978832152e-02,  9.0342861444e-02,
     9.8796103366e-02,  1.0733642163e-01,  1.1596163828e-01,  1.2466953436e-01,
     1.3345785076e-01,  1.4232428890e-01,  1.5126651155e-01,  1.6028214364e-01,
     1.6936877306e-01,  1.7852395152e-01,  1.8774519542e-01,  1.9702998668e-01,
     2.0637577364e-01,  2.1577997202e-01,  2.2523996577e-01,  2.3475310803e-01,
     2.4431672212e-01,  2.5392810244e-01,  2.6358451550e-01,  2.7328320092e-01,
     2.8302137236e-01,  2.9279621866e-01,  3.0260490475e-01,  3.1244457280e-01,
     3.2231234320e-01,  3.3220531566e-01,  3.4212057029e-01,  3.5205516869e-01,
     3.6200615503e-01,  3.7197055716e-01,  3.8194538776e-01,  3.9192764542e-01,
     4.0191431581e-01,  4.1190237277e-01,  4.2188877953e-01,  4.3187048980e-01,
     4.4184444896e-01,  4.5180759522e-01,  4.6175686079e-01,  4.7168917305e-01,
     4.8160145575e-01,  4.9149063014e-01,  5.0135361624e-01,  5.1118733393e-01,
     5.2098870421e-01,  5.3075465040e-01,  5.4048209924e-01,  5.5016798221e-01,
     5.5980923661e-01,  5.6940280682e-01,  5.7894564548e-01,  5.8843471466e-01,
     5.9786698706e-01,  6.0723944718e-01,  6.1654909255e-01,  6.2579293484e-01,
     6.3496800108e-01,  6.4407133482e-01,  6.5309999728e-01,  6.6205106853e-01,
     6.7092164862e-01,  6.7970885873e-01,  6.8840984234e-01,  6.9702176631e-01,
     7.0554182204e-01,  7.1396722657e-01,  7.2229522367e-01,  7.3052308496e-01,
     7.3864811100e-01,  7.4666763233e-01,  7.5457901055e-01,  7.6237963940e-01,
     7.7006694574e-01,  7.7763839062e-01,  7.8509147031e-01,  7.9242371724e-01,
     7.9963270104e-01,  8.0671602951e-01,  8.1367134952e-01,  8.2049634805e-01,
     8.2718875303e-01,  8.3374633430e-01,  8.4016690448e-01,  8.4644831988e-01,
     8.5258848132e-01,  8.5858533503e-01,  8.6443687340e-01,  8.7014113587e-01,
     8.7569620966e-01,  8.8110023057e-01,  8.8635138372e-01,  8.9144790429e-01,
     8.9638807824e-01,  9.0117024296e-01,  9.0579278799e-01,  9.1025415565e-01,
     9.1455284169e-01,  9.1868739584e-01,  9.2265642247e-01,  9.2645858111e-01,
     9.3009258700e-01,  9.3355721161e-01,  9.3685128316e-01,  9.3997368705e-01,
     9.4292336636e-01,  9.4569932224e-01,  9.4830061433e-01,  9.5072636115e-01,
     9.5297574045e-01,  9.5504798952e-01,  9.5694240555e-01,  9.5865834587e-01,
     9.6019522823e-01,  9.6155253103e-01,  9.6272979354e-01,  9.6372661609e-01,
     9.6454266020e-01,  9.6517764877e-01,  9.6563136615e-01,  9.6590365823e-01,
     9.6599443253e-01,  9.6590365823e-01,  9.6563136615e-01,  9.6517764877e-01,
     9.6454266020e-01,  9.6372661609e-01,  9.6272979354e-01,  9.6155253103e-01,
     9.6019522823e-01,  9.5865834587e-01,  9.5694240555e-01,  9.5504798952e-01,
     9.5297574045e-01,  9.5072636115e-01,  9.4830061433e-01,  9.4569932224e-01,
     9.4292336636e-01,  9.3997368705e-01,  9.3685128316e-01,  9.3355721161e-01,
     9.3009258700e-01,  9.2645858111e-01,  9.2265642247e-01,  9.1868739584e-01,
     9.1455284169e-01,  9.1025415565e-01,  9.0579278799e-01,  9.0117024296e-01,
     8.9638807824e-01,  8.9144790429e-01,  8.8635138372e-01,  8.8110023057e-01,
     8.7569620966e-01,  8.7014113587e-01,  8.6443687340e-01,  8.5858533503e-01,
     8.5258848132e-01,  8.4644831988e-01,  8.4016690448e-01,  8.3374633430e-01,
     8.2718875303e-01,  8.2049634805e-01,  8.1367134952e-01,  8.0671602951e-01,
     7.9963270104e-01,  7.9242371724e-01,  7.8509147031e-01,  7.7763839062e-01,
     7.7006694574e-01,  7.6237963940e-01,  7.5457901055e-01,  7.4666763233e-01,
     7.3864811100e-01,  7.3052308496e-01,  7.2229522367e-01,  7.1396722657e-01,
     7.0554182204e-01,  6.9702176631e-01,  6.8840984234e-01,  6.7970885873e-01,
     6.7092164862e-01,  6.6205106853e-01,  6.5309999728e-01,  6.4407133482e-01,
     6.3496800108e-01,  6.2579293484e-01,  6.1654909255e-01,  6.0723944718e-01,
     5.9786698706e-01,  5.8843471466e-01,  5.7894564548e-01,  5.6940280682e-01,
     5.5980923661e-01,  5.5016798221e-01,  5.4048209924e-01,  5.3075465040e-01,
     5.2098870421e-01,  5.1118733393e-01,  5.0135361624e-01,  4.9149063014e-01,
     4.8160145575e-01,  4.7168917305e-01,  4.6175686079e-01,  4.5180759522e-01,
     4.4184444896e-01,  4.3187048980e-01,  4.2188877953e-01,  4.1190237277e-01,
     4.0191431581e-01,  3.9192764542e-01,  3.8194538776e-01,  3.7197055716e-01,
     3.6200615503e-01,  3.5205516869e-01,  3.4212057029e-01,  3.3220531566e-01,
     3.2231234320e-01,  3.1244457280e-01,  3.0260490475e-01,  2.9279621866e-01,
     2.8302137236e-01,  2.7328320092e-01,  2.6358451550e-01,  2.5392810244e-01,
     2.4431672212e-01,  2.3475310803e-01,  2.2523996577e-01,  2.1577997202e-01,
     2.0637577364e-01,  1.9702998668e-01,  1.8774519542e-01,  1.7852395152e-01,
     1.6936877306e-01,  1.6028214364e-01,  1.5126651155e-01,  1.4232428890e-01,
     1.3345785076e-01,  1.2466953436e-01,  1.1596163828e-01,  1.0733642163e-01,
     9.8796103366e-02,  9.0342861444e-02,  8.1978832152e-02,  7.3706109379e-02,
     6.5526743922e-02,  5.7442742818e-02,  4.9456068692e-02,  4.1568639124e-02,
     3.3782326036e-02,  2.6098955109e-02,  1.8520305205e-02,  1.1048107822e-02,
     3.6840465721e-03, -3.5702433330e-03, -1.0713175559e-02, -1.7743213112e-02,
    -2.4658868780e-02, -3.1458705549e-02, -3.8141337000e-02, -4.4705427678e-02,
    -5.1149693447e-02, -5.7472901815e-02, -6.3673872237e-02, -6.9751476402e-02,
    -7.5704638489e-02, -8.1532335403e-02, -8.7233596988e-02, -9.2807506219e-02,
    -9.8253199367e-02, -1.0356986614e-01, -1.0875674982e-01, -1.1381314734e-01,
    -1.1873840935e-01, -1.2353194032e-01, -1.2819319851e-01, -1.3272169599e-01,
    -1.3711699866e-01, -1.4137872616e-01, -1.4550655183e-01, -1.4950020263e-01,
    -1.5335945901e-01, -1.5708415480e-01, -1.6067417705e-01, -1.6412946585e-01,
    -1.6745001412e-01, -1.7063586742e-01, -1.7368712370e-01, -1.7660393299e-01,
    -1.7938649720e-01, -1.8203506974e-01, -1.8454995521e-01, -1.8693150908e-01,
    -1.8918013729e-01, -1.9129629587e-01, -1.9328049053e-01, -1.9513327624e-01,
    -1.9685525675e-01, -1.9844708416e-01, -1.9990945840e-01, -2.0124312672e-01,
    -2.0244888321e-01, -2.0352756818e-01, -2.0448006766e-01, -2.0530731281e-01,
    -2.0601027926e-01, -2.0658998660e-01, -2.0704749764e-01, -2.0738391783e-01,
    -2.0760039456e-01, -2.0769811651e-01, -2.0767831293e-01, -2.0754225290e-01,
    -2.0729124467e-01, -2.0692663486e-01, -2.0644980775e-01, -2.0586218446e-01,
    -2.0516522220e-01, -2.0436041348e-01, -2.0344928527e-01, -2.0243339822e-01,
    -2.0131434579e-01, -2.0009375344e-01, -1.9877327775e-01, -1.9735460555e-01,
    -1.9583945308e-01, -1.9422956508e-01, -1.9252671390e-01, -1.9073269860e-01,
    -1.8884934403e-01, -1.8687849993e-01, -1.8482204000e-01, -1.8268186093e-01,
    -1.8045988153e-01, -1.7815804171e-01, -1.7577830160e-01, -1.7332264052e-01,
    -1.7079305608e-01, -1.6819156318e-01, -1.6552019305e-01, -1.6278099229e-01,
    -1.5997602188e-01, -1.5710735618e-01, -1.5417708199e-01, -1.5118729757e-01,
    -1.4814011161e-01, -1.4503764229e-01, -1.4188201627e-01, -1.3867536772e-01,
    -1.3541983734e-01, -1.3211757138e-01, -1.2877072063e-01, -1.2538143948e-01,
    -1.2195188493e-01, -1.1848421560e-01, -1.1498059078e-01, -1.1144316948e-01,
    -1.0787410942e-01, -1.0427556612e-01, -1.0064969193e-01, -9.6998635069e-02,
    -9.3324538715e-02, -8.9629540047e-02, -8.5915769328e-02, -8.2185348981e-02,
    -7.8440392679e-02, -7.4683004439e-02, -7.0915277727e-02, -6.7139294567e-02,
    -6.3357124664e-02, -5.9570824539e-02, -5.5782436662e-02, -5.1993988609e-02,
    -4.8207492221e-02, -4.4424942777e-02, -4.0648318176e-02, -3.6879578136e-02,
    -3.3120663398e-02, -2.9373494945e-02, -2.5639973237e-02, -2.1921977453e-02,
    -1.8221364752e-02, -1.4539969541e-02, -1.0879602764e-02, -7.2420511994e-03,
    -3.6290767732e-03, -4.2415889919e-05,  3.5162212259e-03,  7.0451511706e-03,
     1.0542717988e-02,  1.4007293779e-02,  1.7437279298e-02,  2.0831104530e-02,
     2.4187229253e-02,  2.7504143583e-02,  3.0780368507e-02,  3.4014456392e-02,
     3.7204991479e-02,  4.0350590364e-02,  4.3449902460e-02,  4.6501610434e-02,
     4.9504430638e-02,  5.2457113515e-02,  5.5358443989e-02,  5.8207241835e-02,
     6.1002362035e-02,  6.3742695112e-02,  6.6427167446e-02,  6.9054741574e-02,
     7.1624416469e-02,  7.4135227804e-02,  7.6586248191e-02,  7.8976587406e-02,
     8.1305392597e-02,  8.3571848469e-02,  8.5775177452e-02,  8.7914639851e-02,
     8.9989533974e-02,  9.1999196249e-02,  9.3943001311e-02,  9.5820362082e-02,
     9.7630729822e-02,  9.9373594172e-02,  1.0104848316e-01,  1.0265496323e-01,
     1.0419263918e-01,  1.0566115416e-01,  1.0706018962e-01,  1.0838946520e-01,
     1.0964873868e-01,  1.1083780584e-01,  1.1195650036e-01,  1.1300469367e-01,
     1.1398229475e-01,  1.1488925002e-01,  1.1572554309e-01,  1.1649119454e-01,
     1.1718626175e-01,  1.1781083859e-01,  1.1836505519e-01,  1.1884907765e-01,
     1.1926310776e-01,  1.1960738263e-01,  1.1988217446e-01,  1.2008779012e-01,
     1.2022457079e-01,  1.2029289166e-01,  1.2029316146e-01,  1.2022582209e-01,
     1.2009134822e-01,  1.1989024684e-01,  1.1962305681e-01,  1.1929034841e-01,
     1.1889272288e-01,  1.1843081192e-01,  1.1790527720e-01,  1.1731680984e-01,
     1.1666612991e-01,  1.1595398588e-01,  1.1518115408e-01,  1.1434843815e-01,
     1.1345666847e-01,  1.1250670158e-01,  1.1149941959e-01,  1.1043572962e-01,
     1.0931656313e-01,  1.0814287536e-01,  1.0691564468e-01,  1.0563587196e-01,
     1.0430457994e-01,  1.0292281257e-01,  1.0149163438e-01,  1.0001212976e-01,
     9.8485402366e-02,  9.6912574388e-02,  9.5294785892e-02,  9.3633194128e-02,
     9.1928972841e-02,  9.0183311569e-02,  8.8397414947e-02,  8.6572501997e-02,
     8.4709805420e-02,  8.2810570882e-02,  8.0876056301e-02,  7.8907531125e-02,
     7.6906275613e-02,  7.4873580114e-02,  7.2810744337e-02,  7.0719076632e-02,
     6.8599893258e-02,  6.6454517662e-02,  6.4284279745e-02,  6.2090515144e-02,
     5.9874564497e-02,  5.7637772724e-02,  5.5381488303e-02,  5.3107062544e-02,
     5.0815848874e-02,  4.8509202116e-02,  4.6188477777e-02,  4.3855031333e-02,
     4.1510217527e-02,  3.9155389655e-02,  3.6791898877e-02,  3.4421093512e-02,
     3.2044318353e-02,  2.9662913978e-02,  2.7278216073e-02,  2.4891554752e-02,
     2.2504253895e-02,  2.0117630481e-02,  1.7732993936e-02,  1.5351645482e-02,
     1.2974877497e-02,  1.0603972885e-02,  8.2402044434e-03,  5.8848342528e-03,
     3.5391130636e-03,  1.2042796983e-03, -1.1184395412e-03, -3.4278314541e-03,
    -5.7226965139e-03, -8.0018494307e-03, -1.0264119704e-02, -1.2508352163e-02,
    -1.4733407503e-02, -1.6938162802e-02, -1.9121512031e-02, -2.1282366558e-02,
    -2.3419655629e-02, -2.5532326848e-02, -2.7619346642e-02, -2.9679700709e-02,
    -3.1712394465e-02, -3.3716453465e-02, -3.5690923824e-02, -3.7634872617e-02,
    -3.9547388269e-02, -4.1427580933e-02, -4.3274582856e-02, -4.5087548726e-02,
    -4.6865656014e-02, -4.8608105295e-02, -5.0314120558e-02, -5.1982949506e-02,
    -5.3613863840e-02, -5.5206159523e-02, -5.6759157041e-02, -5.8272201642e-02,
    -5.9744663563e-02, -6.1175938245e-02, -6.2565446533e-02, -6.3912634857e-02,
    -6.5216975406e-02, -6.6477966285e-02, -6.7695131656e-02, -6.8868021863e-02,
    -6.9996213552e-02, -7.1079309764e-02, -7.2116940024e-02, -7.3108760412e-02,
    -7.4054453614e-02, -7.4953728970e-02, -7.5806322501e-02, -7.6611996920e-02,
    -7.7370541634e-02, -7.8081772729e-02, -7.8745532941e-02, -7.9361691617e-02,
    -7.9930144654e-02, -8.0450814434e-02, -8.0923649735e-02, -8.1348625641e-02,
    -8.1725743426e-02, -8.2055030432e-02, -8.2336539933e-02, -8.2570350984e-02,
    -8.2756568258e-02, -8.2895321870e-02, -8.2986767190e-02, -8.3031084639e-02,
    -8.3028479480e-02, -8.2979181588e-02, -8.2883445214e-02, -8.2741548737e-02,
    -8.2553794402e-02, -8.2320508047e-02, -8.2042038817e-02, -8.1718758875e-02,
    -8.1351063088e-02, -8.0939368718e-02, -8.0484115088e-02, -7.9985763251e-02,
    -7.9444795637e-02, -7.8861715699e-02, -7.8237047544e-02, -7.7571335560e-02,
    -7.6865144025e-02, -7.6119056718e-02, -7.5333676511e-02, -7.4509624964e-02,
    -7.3647541898e-02, -7.2748084971e-02, -7.1811929246e-02, -7.0839766740e-02,
    -6.9832305984e-02, -6.8790271558e-02, -6.7714403634e-02, -6.6605457502e-02,
    -6.5464203097e-02, -6.4291424515e-02, -6.3087919531e-02, -6.1854499102e-02,
    -6.0591986872e-02, -5.9301218669e-02, -5.7983042000e-02, -5.6638315543e-02,
    -5.5267908628e-02, -5.3872700723e-02, -5.2453580912e-02, -5.1011447372e-02,
    -4.9547206846e-02, -4.8061774116e-02, -4.6556071467e-02, -4.5031028162e-02,
    -4.3487579902e-02, -4.1926668294e-02, -4.0349240312e-02, -3.8756247763e-02,
    -3.7148646747e-02, -3.5527397123e-02, -3.3893461969e-02, -3.2247807047e-02,
    -3.0591400266e-02, -2.8925211150e-02, -2.7250210301e-02, -2.5567368871e-02,
    -2.3877658030e-02, -2.2182048438e-02, -2.0481509723e-02, -1.8777009954e-02,
    -1.7069515125e-02, -1.5359988639e-02, -1.3649390792e-02, -1.1938678268e-02,
    -1.0228803633e-02, -8.5207148330e-03, -6.8153546992e-03, -5.1136604573e-03,
    -3.4165632398e-03, -1.7249876071e-03, -3.9851069993e-05,  1.6379363784e-03,
     3.3074727272e-03,  4.9678644051e-03,  6.6182267312e-03,  8.2576843598e-03,
     9.8853717179e-03,  1.1500433436e-02,  1.3102024771e-02,  1.4689312023e-02,
     1.6261472942e-02,  1.7817697131e-02,  1.9357186434e-02,  2.0879155326e-02,
     2.2382831284e-02,  2.3867455156e-02,  2.5332281521e-02,  2.6776579038e-02,
     2.8199630784e-02,  2.9600734589e-02,  3.0979203359e-02,  3.2334365386e-02,
     3.3665564650e-02,  3.4972161119e-02,  3.6253531027e-02,  3.7509067149e-02,
     3.8738179066e-02,  3.9940293419e-02,  4.1114854149e-02,  4.2261322736e-02,
     4.3379178415e-02,  4.4467918394e-02,  4.5527058052e-02,  4.6556131130e-02,
     4.7554689914e-02,  4.8522305400e-02,  4.9458567459e-02,  5.0363084978e-02,
     5.1235486000e-02,  5.2075417853e-02,  5.2882547258e-02,  5.3656560441e-02,
     5.4397163219e-02,  5.5104081086e-02,  5.5777059284e-02,  5.6415862861e-02,
     5.7020276723e-02,  5.7590105670e-02,  5.8125174421e-02,  5.8625327636e-02,
     5.9090429917e-02,  5.9520365803e-02,  5.9915039753e-02,  6.0274376124e-02,
     6.0598319124e-02,  6.0886832773e-02,  6.1139900837e-02,  6.1357526761e-02,
     6.1539733590e-02,  6.1686563877e-02,  6.1798079584e-02,  6.1874361967e-02,
     6.1915511462e-02,  6.1921647547e-02,  6.1892908605e-02,  6.1829451776e-02,
     6.1731452789e-02,  6.1599105801e-02,  6.1432623212e-02,  6.1232235479e-02,
     6.0998190920e-02,  6.0730755502e-02,  6.0430212634e-02,  6.0096862935e-02,
     5.9731024008e-02,  5.9333030196e-02,  5.8903232330e-02,  5.8441997479e-02,
     5.7949708679e-02,  5.7426764663e-02,  5.6873579580e-02,  5.6290582708e-02,
     5.5678218157e-02,  5.5036944572e-02,  5.4367234819e-02,  5.3669575673e-02,
     5.2944467495e-02,  5.2192423905e-02,  5.1413971450e-02,  5.0609649261e-02,
     4.9780008707e-02,  4.8925613051e-02,  4.8047037086e-02,  4.7144866782e-02,
     4.6219698913e-02,  4.5272140691e-02,  4.4302809394e-02,  4.3312331982e-02,
     4.2301344717e-02,  4.1270492778e-02,  4.0220429871e-02,  3.9151817837e-02,
     3.8065326252e-02,  3.6961632035e-02,  3.5841419041e-02,  3.4705377664e-02,
     3.3554204425e-02,  3.2388601569e-02,  3.1209276656e-02,  3.0016942147e-02,
     2.8812314997e-02,  2.7596116242e-02,  2.6369070581e-02,  2.5131905971e-02,
     2.3885353203e-02,  2.2630145498e-02,  2.1367018085e-02,  2.0096707794e-02,
     1.8819952640e-02,  1.7537491410e-02,  1.6250063257e-02,  1.4958407286e-02,
     1.3663262150e-02,  1.2365365639e-02,  1.1065454280e-02,  9.7642629355e-03,
     8.4625243991e-03,  7.1609690030e-03,  5.8603242218e-03,  4.5613142820e-03,
     3.2646597732e-03,  1.9710772644e-03,  6.8127892215e-04, -6.0402786624e-04,
    -1.8841408659e-03, -3.1583633634e-03, -4.4260045317e-03, -5.6863797901e-03,
    -6.9388111592e-03, -8.1826276123e-03, -9.4171654199e-03, -1.0641768490e-02,
    -1.1855788704e-02, -1.3058586243e-02, -1.4249529914e-02, -1.5427997466e-02,
    -1.6593375902e-02, -1.7745061784e-02, -1.8882461530e-02, -2.0004991711e-02,
    -2.1112079330e-02, -2.2203162108e-02, -2.3277688751e-02, -2.4335119217e-02,
    -2.5374924973e-02, -2.6396589247e-02, -2.7399607270e-02, -2.8383486514e-02,
    -2.9347746916e-02, -3.0291921101e-02, -3.1215554598e-02, -3.2118206037e-02,
    -3.2999447355e-02, -3.3858863977e-02, -3.4696055003e-02, -3.5510633376e-02,
    -3.6302226050e-02, -3.7070474142e-02, -3.7815033086e-02, -3.8535572764e-02,
    -3.9231777646e-02, -3.9903346903e-02, -4.0549994530e-02, -4.1171449444e-02,
    -4.1767455586e-02, -4.2337772002e-02, -4.2882172933e-02, -4.3400447874e-02,
    -4.3892401646e-02, -4.4357854443e-02, -4.4796641879e-02, -4.5208615024e-02,
    -4.5593640432e-02, -4.5951600158e-02, -4.6282391770e-02, -4.6585928350e-02,
    -4.6862138486e-02, -4.7110966257e-02, -4.7332371210e-02, -4.7526328328e-02,
    -4.7692827984e-02, -4.7831875898e-02, -4.7943493077e-02, -4.8027715749e-02,
    -4.8084595288e-02, -4.8114198134e-02, -4.8116605701e-02, -4.8091914281e-02,
    -4.8040234936e-02, -4.7961693384e-02, -4.7856429880e-02, -4.7724599086e-02,
    -4.7566369932e-02, -4.7381925474e-02, -4.7171462745e-02, -4.6935192590e-02,
    -4.6673339508e-02, -4.6386141473e-02, -4.6073849760e-02, -4.5736728754e-02,
    -4.5375055760e-02, -4.4989120803e-02, -4.4579226422e-02, -4.4145687460e-02,
    -4.3688830843e-02, -4.3208995359e-02, -4.2706531427e-02, -4.2181800862e-02,
    -4.1635176632e-02, -4.1067042618e-02, -4.0477793356e-02, -3.9867833783e-02,
    -3.9237578977e-02, -3.8587453890e-02, -3.7917893077e-02, -3.7229340422e-02,
    -3.6522248860e-02, -3.5797080089e-02, -3.5054304290e-02, -3.4294399828e-02,
    -3.3517852966e-02, -3.2725157562e-02, -3.1916814770e-02, -3.1093332733e-02,
    -3.0255226281e-02, -2.9403016621e-02, -2.8537231019e-02, -2.7658402493e-02,
    -2.6767069495e-02, -2.5863775589e-02, -2.4949069134e-02, -2.4023502965e-02,
    -2.3087634062e-02, -2.2142023236e-02, -2.1187234795e-02, -2.0223836223e-02,
    -1.9252397852e-02, -1.8273492534e-02, -1.7287695314e-02, -1.6295583104e-02,
    -1.5297734354e-02, -1.4294728722e-02, -1.3287146753e-02, -1.2275569548e-02,
    -1.1260578442e-02, -1.0242754676e-02, -9.2226790761e-03, -8.2009317312e-03,
    -7.1780916719e-03, -6.1547365516e-03, -5.1314423297e-03, -4.1087829572e-03,
    -3.0873300626e-03, -2.0676526422e-03, -1.0503167520e-03, -3.5885202143e-05,
     9.7508274565e-04,  1.9820316773e-03,  2.9844103231e-03,  3.9816718507e-03,
     4.9732741549e-03,  5.9586801435e-03,  6.9373580202e-03,  7.9087815622e-03,
     8.8724303956e-03,  9.8277902647e-03,  1.0774353299e-02,  1.1711618274e-02,
     1.2639090868e-02,  1.3556283917e-02,  1.4462717656e-02,  1.5357919970e-02,
     1.6241426625e-02,  1.7112781501e-02,  1.7971536824e-02,  1.8817253381e-02,
     1.9649500740e-02,  2.0467857462e-02,  2.1271911301e-02,  2.2061259407e-02,
     2.2835508519e-02,  2.3594275150e-02,  2.4337185769e-02,  2.5063876979e-02,
     2.5773995681e-02,  2.6467199240e-02,  2.7143155639e-02,  2.7801543632e-02,
     2.8442052883e-02,  2.9064384107e-02,  2.9668249197e-02,  3.0253371348e-02,
     3.0819485176e-02,  3.1366336826e-02,  3.1893684075e-02,  3.2401296431e-02,
     3.2888955218e-02,  3.3356453666e-02,  3.3803596980e-02,  3.4230202412e-02,
     3.4636099322e-02,  3.5021129237e-02,  3.5385145891e-02,  3.5728015275e-02,
     3.6049615667e-02,  3.6349837657e-02,  3.6628584174e-02,  3.6885770491e-02,
     3.7121324241e-02,  3.7335185406e-02,  3.7527306321e-02,  3.7697651648e-02,
     3.7846198367e-02,  3.7972935738e-02,  3.8077865273e-02,  3.8161000693e-02,
     3.8222367879e-02,  3.8262004817e-02,  3.8279961542e-02,  3.8276300063e-02,
     3.8251094294e-02,  3.8204429974e-02,  3.8136404576e-02,  3.8047127218e-02,
     3.7936718561e-02,  3.7805310707e-02,  3.7653047085e-02,  3.7480082334e-02,
     3.7286582181e-02,  3.7072723311e-02,  3.6838693233e-02,  3.6584690139e-02,
     3.6310922761e-02,  3.6017610214e-02,  3.5704981849e-02,  3.5373277079e-02,
     3.5022745225e-02,  3.4653645336e-02,  3.4266246013e-02,  3.3860825233e-02,
     3.3437670157e-02,  3.2997076944e-02,  3.2539350553e-02,  3.2064804544e-02,
     3.1573760879e-02,  3.1066549709e-02,  3.0543509166e-02,  3.0004985145e-02,
     2.9451331092e-02,  2.8882907774e-02,  2.8300083057e-02,  2.7703231679e-02,
     2.7092735013e-02,  2.6468980839e-02,  2.5832363101e-02,  2.5183281667e-02,
     2.4522142090e-02,  2.3849355360e-02,  2.3165337655e-02,  2.2470510096e-02,
     2.1765298490e-02,  2.1050133080e-02,  2.0325448291e-02,  1.9591682470e-02,
     1.8849277631e-02,  1.8098679193e-02,  1.7340335724e-02,  1.6574698675e-02,
     1.5802222123e-02,  1.5023362502e-02,  1.4238578348e-02,  1.3448330030e-02,
     1.2653079488e-02,  1.1853289969e-02,  1.1049425764e-02,  1.0241951947e-02,
     9.4313341068e-03,  8.6180380906e-03,  7.8025297386e-03,  6.9852746248e-03,
     6.1667377968e-03,  5.3473835179e-03,  4.5276750085e-03,  3.7080741913e-03,
     2.8890414367e-03,  2.0710353092e-03,  1.2545123175e-03,  4.3992666514e-04,
    -3.7226999674e-04, -1.1816288136e-03, -1.9877039697e-03, -2.7900529290e-03,
    -3.5882366713e-03, -4.3818199271e-03, -5.1703714102e-03, -5.9534640450e-03,
    -6.7306751936e-03, -7.5015868771e-03, -8.2657859957e-03, -9.0228645441e-03,
    -9.7724198238e-03, -1.0514054651e-02, -1.1247377564e-02, -1.1972003019e-02,
    -1.2687551594e-02, -1.3393650176e-02, -1.4089932153e-02, -1.4776037597e-02,
    -1.5451613447e-02, -1.6116313682e-02, -1.6769799492e-02, -1.7411739449e-02,
    -1.8041809664e-02, -1.8659693949e-02, -1.9265083966e-02, -1.9857679379e-02,
    -2.0437187991e-02, -2.1003325888e-02, -2.1555817566e-02, -2.2094396065e-02,
    -2.2618803085e-02, -2.3128789109e-02, -2.3624113510e-02, -2.4104544660e-02,
    -2.4569860033e-02, -2.5019846298e-02, -2.5454299411e-02, -2.5873024697e-02,
    -2.6275836935e-02, -2.6662560426e-02, -2.7033029066e-02, -2.7387086406e-02,
    -2.7724585710e-02, -2.8045390005e-02, -2.8349372130e-02, -2.8636414773e-02,
    -2.8906410509e-02, -2.9159261826e-02, -2.9394881148e-02, -2.9613190858e-02,
    -2.9814123301e-02, -2.9997620801e-02, -3.0163635652e-02, -3.0312130120e-02,
    -3.0443076431e-02, -3.0556456753e-02, -3.0652263177e-02, -3.0730497690e-02,
    -3.0791172140e-02, -3.0834308202e-02, -3.0859937334e-02, -3.0868100725e-02,
    -3.0858849247e-02, -3.0832243391e-02, -3.0788353206e-02, -3.0727258226e-02,
    -3.0649047400e-02, -3.0553819011e-02, -3.0441680589e-02, -3.0312748824e-02,
    -3.0167149472e-02, -3.0005017255e-02, -2.9826495756e-02, -2.9631737314e-02,
    -2.9420902908e-02, -2.9194162041e-02, -2.8951692618e-02, -2.8693680818e-02,
    -2.8420320966e-02, -2.8131815399e-02, -2.7828374324e-02, -2.7510215678e-02,
    -2.7177564980e-02, -2.6830655183e-02, -2.6469726516e-02, -2.6095026330e-02,
    -2.5706808936e-02, -2.5305335439e-02, -2.4890873573e-02, -2.4463697527e-02,
    -2.4024087770e-02, -2.3572330881e-02, -2.3108719360e-02, -2.2633551452e-02,
    -2.2147130956e-02, -2.1649767044e-02, -2.1141774063e-02, -2.0623471349e-02,
    -2.0095183029e-02, -1.9557237822e-02, -1.9009968845e-02, -1.8453713409e-02,
    -1.7888812818e-02, -1.7315612161e-02, -1.6734460114e-02, -1.6145708727e-02,
    -1.5549713216e-02, -1.4946831758e-02, -1.4337425274e-02, -1.3721857227e-02,
    -1.3100493399e-02, -1.2473701688e-02, -1.1841851888e-02, -1.1205315482e-02,
    -1.0564465420e-02, -9.9196759124e-03, -9.2713222117e-03, -8.6197804001e-03,
    -7.9654271752e-03, -7.3086396367e-03, -6.6497950734e-03, -5.9892707506e-03,
    -5.3274436982e-03, -4.6646904999e-03, -4.0013870832e-03, -3.3379085095e-03,
    -2.6746287673e-03, -2.0119205646e-03, -1.3501551235e-03, -6.8970197709e-04,
    -3.0928766166e-05,  6.2579896088e-04,  1.2801179467e-03,  1.9316674227e-03,
     2.5800893034e-03,  3.2250283792e-03,  3.8661325076e-03,  4.5030528007e-03,
     5.1354438119e-03,  5.7629637189e-03,  6.3852745044e-03,  7.0020421346e-03,
     7.6129367342e-03,  8.2176327591e-03,  8.8158091654e-03,  9.4071495767e-03,
     9.9913424465e-03,  1.0568081219e-02,  1.1137064485e-02,  1.1697996137e-02,
     1.2250585517e-02,  1.2794547566e-02,  1.3329602961e-02,  1.3855478263e-02,
     1.4371906043e-02,  1.4878625023e-02,  1.5375380194e-02,  1.5861922949e-02,
     1.6338011198e-02,  1.6803409483e-02,  1.7257889093e-02,  1.7701228171e-02,
     1.8133211815e-02,  1.8553632181e-02,  1.8962288574e-02,  1.9358987543e-02,
     1.9743542964e-02,  2.0115776123e-02,  2.0475515794e-02,  2.0822598311e-02,
     2.1156867639e-02,  2.1478175434e-02,  2.1786381109e-02,  2.2081351881e-02,
     2.2362962830e-02,  2.2631096936e-02,  2.2885645130e-02,  2.3126506322e-02,
     2.3353587440e-02,  2.3566803452e-02,  2.3766077394e-02,  2.3951340385e-02,
     2.4122531641e-02,  2.4279598487e-02,  2.4422496357e-02,  2.4551188799e-02,
     2.4665647467e-02,  2.4765852113e-02,  2.4851790573e-02,  2.4923458750e-02,
     2.4980860591e-02,  2.5024008058e-02,  2.5052921100e-02,  2.5067627614e-02,
     2.5068163409e-02,  2.5054572154e-02,  2.5026905338e-02,  2.4985222211e-02,
     2.4929589729e-02,  2.4860082491e-02,  2.4776782676e-02,  2.4679779971e-02,
     2.4569171500e-02,  2.4445061742e-02,  2.4307562455e-02,  2.4156792588e-02,
     2.3992878190e-02,  2.3815952322e-02,  2.3626154955e-02,  2.3423632875e-02,
     2.3208539577e-02,  2.2981035157e-02,  2.2741286203e-02,  2.2489465683e-02,
     2.2225752823e-02,  2.1950332993e-02,  2.1663397578e-02,  2.1365143858e-02,
     2.1055774872e-02,  2.0735499292e-02,  2.0404531284e-02,  2.0063090373e-02,
     1.9711401303e-02,  1.9349693891e-02,  1.8978202889e-02,  1.8597167830e-02,
     1.8206832880e-02,  1.7807446689e-02,  1.7399262236e-02,  1.6982536671e-02,
     1.6557531161e-02,  1.6124510728e-02,  1.5683744090e-02,  1.5235503497e-02,
     1.4780064566e-02,  1.4317706119e-02,  1.3848710013e-02,  1.3373360971e-02,
     1.2891946419e-02,  1.2404756310e-02,  1.1912082952e-02,  1.1414220844e-02,
     1.0911466495e-02,  1.0404118256e-02,  9.8924761442e-03,  9.3768416703e-03,
     8.8575176630e-03,  8.3348080951e-03,  7.8090179086e-03,  7.2804528401e-03,
     6.7494192468e-03,  6.2162239309e-03,  5.6811739666e-03,  5.1445765260e-03,
     4.6067387054e-03,  4.0679673534e-03,  3.5285688986e-03,  2.9888491777e-03,
     2.4491132664e-03,  1.9096653087e-03,  1.3708083491e-03,  8.3284416544e-04,
     2.9607310255e-04, -2.3920609248e-04, -7.7269643332e-04, -1.3041028564e-03,
    -1.8331323817e-03, -2.3594942703e-03, -2.8829001820e-03, -3.4030643303e-03,
    -3.9197036349e-03, -4.4325378739e-03, -4.9412898318e-03, -5.4456854470e-03,
    -5.9454539561e-03, -6.4403280370e-03, -6.9300439476e-03, -7.4143416648e-03,
    -7.8929650188e-03, -8.3656618254e-03, -8.8321840164e-03, -9.2922877663e-03,
    -9.7457336164e-03, -1.0192286597e-02, -1.0631716346e-02, -1.1063797223e-02,
    -1.1488308423e-02, -1.1905034088e-02, -1.2313763409e-02, -1.2714290734e-02,
    -1.3106415663e-02, -1.3489943149e-02, -1.3864683591e-02, -1.4230452921e-02,
    -1.4587072693e-02, -1.4934370169e-02, -1.5272178392e-02, -1.5600336268e-02,
    -1.5918688638e-02, -1.6227086344e-02, -1.6525386298e-02, -1.6813451543e-02,
    -1.7091151312e-02, -1.7358361079e-02, -1.7614962616e-02, -1.7860844035e-02,
    -1.8095899836e-02, -1.8320030942e-02, -1.8533144741e-02, -1.8735155111e-02,
    -1.8925982457e-02, -1.9105553729e-02, -1.9273802446e-02, -1.9430668713e-02,
    -1.9576099236e-02, -1.9710047327e-02, -1.9832472918e-02, -1.9943342553e-02,
    -2.0042629398e-02, -2.0130313225e-02, -2.0206380412e-02, -2.0270823923e-02,
    -2.0323643298e-02, -2.0364844629e-02, -2.0394440539e-02, -2.0412450152e-02,
    -2.0418899065e-02, -2.0413819313e-02, -2.0397249329e-02, -2.0369233906e-02,
    -2.0329824151e-02, -2.0279077436e-02, -2.0217057346e-02, -2.0143833627e-02,
    -2.0059482125e-02, -1.9964084725e-02, -1.9857729287e-02, -1.9740509577e-02,
    -1.9612525199e-02, -1.9473881516e-02, -1.9324689578e-02, -1.9165066037e-02,
    -1.8995133070e-02, -1.8815018286e-02, -1.8624854645e-02, -1.8424780359e-02,
    -1.8214938805e-02, -1.7995478423e-02, -1.7766552619e-02, -1.7528319663e-02,
    -1.7280942586e-02, -1.7024589071e-02, -1.6759431346e-02, -1.6485646073e-02,
    -1.6203414232e-02, -1.5912921009e-02, -1.5614355680e-02, -1.5307911484e-02,
    -1.4993785513e-02, -1.4672178578e-02, -1.4343295091e-02, -1.4007342939e-02,
    -1.3664533352e-02, -1.3315080777e-02, -1.2959202744e-02, -1.2597119738e-02,
    -1.2229055062e-02, -1.1855234704e-02, -1.1475887201e-02, -1.1091243504e-02,
    -1.0701536835e-02, -1.0307002556e-02, -9.9078780229e-03, -9.5044024497e-03,
    -9.0968167665e-03, -8.6853634778e-03, -8.2702865221e-03, -7.8518311288e-03,
    -7.4302436765e-03, -7.0057715502e-03, -6.5786629990e-03, -6.1491669928e-03,
    -5.7175330804e-03, -5.2840112465e-03, -4.8488517693e-03, -4.4123050786e-03,
    -3.9746216138e-03, -3.5360516830e-03, -3.0968453219e-03, -2.6572521536e-03,
    -2.2175212489e-03, -1.7779009881e-03, -1.3386389221e-03, -8.9998163592e-04,
    -4.6217461238e-04, -2.5462096983e-05,  4.0991303625e-04,  8.4370941704e-04,
     1.2756873114e-03,  1.7056087507e-03,  2.1332376614e-03,  2.5583399921e-03,
     2.9806838390e-03,  3.4000395709e-03,  3.8161799513e-03,  4.2288802593e-03,
     4.6379184087e-03,  5.0430750651e-03,  5.4441337616e-03,  5.8408810113e-03,
     6.2331064197e-03,  6.6206027929e-03,  7.0031662453e-03,  7.3805963043e-03,
     7.7526960125e-03,  8.1192720288e-03,  8.4801347257e-03,  8.8350982852e-03,
     9.1839807920e-03,  9.5266043239e-03,  9.8627950398e-03,  1.0192383266e-02,
     1.0515203578e-02,  1.0831094881e-02,  1.1139900488e-02,  1.1441468197e-02,
     1.1735650356e-02,  1.2022303940e-02,  1.2301290612e-02,  1.2572476793e-02,
     1.2835733713e-02,  1.3090937479e-02,  1.3337969123e-02,  1.3576714657e-02,
     1.3807065121e-02,  1.4028916631e-02,  1.4242170417e-02,  1.4446732869e-02,
     1.4642515570e-02,  1.4829435331e-02,  1.5007414222e-02,  1.5176379600e-02,
     1.5336264132e-02,  1.5487005820e-02,  1.5628548014e-02,  1.5760839435e-02,
     1.5883834181e-02,  1.5997491737e-02,  1.6101776987e-02,  1.6196660208e-02,
     1.6282117080e-02,  1.6358128673e-02,  1.6424681450e-02,  1.6481767251e-02,
     1.6529383288e-02,  1.6567532121e-02,  1.6596221649e-02,  1.6615465084e-02,
     1.6625280926e-02,  1.6625692942e-02,  1.6616730129e-02,  1.6598426688e-02,
     1.6570821984e-02,  1.6533960510e-02,  1.6487891845e-02,  1.6432670611e-02,
     1.6368356426e-02,  1.6295013856e-02,  1.6212712361e-02,  1.6121526244e-02,
     1.6021534590e-02,  1.5912821209e-02,  1.5795474573e-02,  1.5669587755e-02,
     1.5535258355e-02,  1.5392588439e-02,  1.5241684462e-02,  1.5082657196e-02,
     1.4915621654e-02,  1.4740697013e-02,  1.4558006531e-02,  1.4367677467e-02,
     1.4169840997e-02,  1.3964632128e-02,  1.3752189604e-02,  1.3532655827e-02,
     1.3306176754e-02,  1.3072901813e-02,  1.2832983798e-02,  1.2586578784e-02,
     1.2333846017e-02,  1.2074947822e-02,  1.1810049499e-02,  1.1539319221e-02,
     1.1262927929e-02,  1.0981049227e-02,  1.0693859278e-02,  1.0401536693e-02,
     1.0104262426e-02,  9.8022196614e-03,  9.4955937054e-03,  9.1845718742e-03,
     8.8693433822e-03,  8.5500992288e-03,  8.2270320852e-03,  7.9003361801e-03,
     7.5702071859e-03,  7.2368421023e-03,  6.9004391423e-03,  6.5611976156e-03,
     6.2193178124e-03,  5.8750008881e-03,  5.5284487461e-03,  5.1798639217e-03,
     4.8294494659e-03,  4.4774088287e-03,  4.1239457432e-03,  3.7692641091e-03,
     3.4135678779e-03,  3.0570609365e-03,  2.6999469929e-03,  2.3424294613e-03,
     1.9847113481e-03,  1.6269951385e-03,  1.2694826838e-03,  9.1237508905e-04,
     5.5587260220e-04,  2.0017450330e-04, -1.5452100546e-04, -5.0801690751e-04,
    -8.6011748035e-04, -1.2106284019e-03, -1.5593568553e-03, -1.9061116343e-03,
    -2.2507032454e-03, -2.5929440101e-03, -2.9326481651e-03, -3.2696319612e-03,
    -3.6037137611e-03, -3.9347141352e-03, -4.2624559565e-03, -4.5867644929e-03,
    -4.9074674993e-03, -5.2243953063e-03, -5.5373809088e-03, -5.8462600517e-03,
    -6.1508713138e-03, -6.4510561910e-03, -6.7466591761e-03, -7.0375278378e-03,
    -7.3235128965e-03, -7.6044682999e-03, -7.8802512945e-03, -8.1507224961e-03,
    -8.4157459589e-03, -8.6751892405e-03, -8.9289234663e-03, -9.1768233914e-03,
    -9.4187674593e-03, -9.6546378600e-03, -9.8843205840e-03, -1.0107705476e-02,
    -1.0324686282e-02, -1.0535160703e-02, -1.0739030434e-02, -1.0936201212e-02,
    -1.1126582852e-02, -1.1310089291e-02, -1.1486638617e-02, -1.1656153109e-02,
    -1.1818559263e-02, -1.1973787820e-02, -1.2121773798e-02, -1.2262456508e-02,
    -1.2395779577e-02, -1.2521690971e-02, -1.2640143003e-02, -1.2751092353e-02,
    -1.2854500074e-02, -1.2950331603e-02, -1.3038556768e-02, -1.3119149787e-02,
    -1.3192089270e-02, -1.3257358223e-02, -1.3314944035e-02, -1.3364838476e-02,
    -1.3407037690e-02, -1.3441542179e-02, -1.3468356790e-02, -1.3487490700e-02,
    -1.3498957399e-02, -1.3502774662e-02, -1.3498964533e-02, -1.3487553296e-02,
    -1.3468571444e-02, -1.3442053654e-02, -1.3408038750e-02, -1.3366569670e-02,
    -1.3317693428e-02, -1.3261461073e-02, -1.3197927652e-02, -1.3127152161e-02,
    -1.3049197503e-02, -1.2964130438e-02, -1.2872021533e-02, -1.2772945112e-02,
    -1.2666979200e-02, -1.2554205468e-02, -1.2434709177e-02, -1.2308579114e-02,
    -1.2175907536e-02, -1.2036790102e-02, -1.1891325812e-02, -1.1739616940e-02,
    -1.1581768960e-02, -1.1417890487e-02, -1.1248093194e-02, -1.1072491749e-02,
    -1.0891203734e-02, -1.0704349572e-02, -1.0512052450e-02, -1.0314438239e-02,
    -1.0111635416e-02, -9.9037749839e-03, -9.6909903854e-03, -9.4734174237e-03,
    -9.2511941766e-03, -9.0244609117e-03, -8.7933599993e-03, -8.5580358260e-03,
    -8.3186347062e-03, -8.0753047934e-03, -7.8281959906e-03, -7.5774598599e-03,
    -7.3232495311e-03, -7.0657196106e-03, -6.8050260892e-03, -6.5413262490e-03,
    -6.2747785710e-03, -6.0055426411e-03, -5.7337790563e-03, -5.4596493312e-03,
    -5.1833158029e-03, -4.9049415370e-03, -4.6246902330e-03, -4.3427261298e-03,
    -4.0592139104e-03, -3.7743186083e-03, -3.4882055122e-03, -3.2010400716e-03,
    -2.9129878038e-03, -2.6242141984e-03, -2.3348846253e-03, -2.0451642403e-03,
    -1.7552178930e-03, -1.4652100342e-03, -1.1753046243e-03, -8.8566504201e-04,
    -5.9645399359e-04, -3.0783342341e-04, -1.9964423943e-05,  2.6699285204e-04,
     5.5287927910e-04,  8.3753684658e-04,  1.1208087435e-03,  1.4025394441e-03,
     1.6825747918e-03,  1.9607620818e-03,  2.2369501429e-03,  2.5109894183e-03,
     2.7827320450e-03,  3.0520319318e-03,  3.3187448370e-03,  3.5827284435e-03,
     3.8438424334e-03,  4.1019485613e-03,  4.3569107253e-03,  4.6085950376e-03,
     4.8568698933e-03,  5.1016060373e-03,  5.3426766300e-03,  5.5799573117e-03,
     5.8133262643e-03,  6.0426642729e-03,  6.2678547847e-03,  6.4887839657e-03,
     6.7053407580e-03,  6.9174169323e-03,  7.1249071407e-03,  7.3277089671e-03,
     7.5257229756e-03,  7.7188527573e-03,  7.9070049755e-03,  8.0900894082e-03,
     8.2680189891e-03,  8.4407098475e-03,  8.6080813445e-03,  8.7700561089e-03,
     8.9265600708e-03,  9.0775224920e-03,  9.2228759961e-03,  9.3625565957e-03,
     9.4965037174e-03,  9.6246602253e-03,  9.7469724424e-03,  9.8633901700e-03,
     9.9738667042e-03,  1.0078358852e-02,  1.0176826945e-02,  1.0269234846e-02,
     1.0355549967e-02,  1.0435743266e-02,  1.0509789259e-02,  1.0577666022e-02,
     1.0639355189e-02,  1.0694841953e-02,  1.0744115062e-02,  1.0787166817e-02,
     1.0823993059e-02,  1.0854593166e-02,  1.0878970039e-02,  1.0897130087e-02,
     1.0909083217e-02,  1.0914842813e-02,  1.0914425719e-02,  1.0907852218e-02,
     1.0895146008e-02,  1.0876334180e-02,  1.0851447192e-02,  1.0820518835e-02,
     1.0783586211e-02,  1.0740689698e-02,  1.0691872913e-02,  1.0637182684e-02,
     1.0576669005e-02,  1.0510385006e-02,  1.0438386904e-02,  1.0360733969e-02,
     1.0277488475e-02,  1.0188715657e-02,  1.0094483666e-02,  9.9948635184e-03,
     9.8899290469e-03,  9.7797568510e-03,  9.6644262436e-03,  9.5440191977e-03,
     9.4186202905e-03,  9.2883166480e-03,  9.1531978871e-03,  9.0133560563e-03,
     8.8688855764e-03,  8.7198831787e-03,  8.5664478428e-03,  8.4086807338e-03,
     8.2466851370e-03,  8.0805663928e-03,  7.9104318311e-03,  7.7363907031e-03,
     7.5585541131e-03,  7.3770349505e-03,  7.1919478191e-03,  7.0034089668e-03,
     6.8115362148e-03,  6.6164488854e-03,  6.4182677295e-03,  6.2171148536e-03,
     6.0131136462e-03,  5.8063887037e-03,  5.5970657561e-03,  5.3852715914e-03,
     5.1711339813e-03,  4.9547816051e-03,  4.7363439733e-03,  4.5159513525e-03,
     4.2937346885e-03,  4.0698255300e-03,  3.8443559523e-03,  3.6174584808e-03,
     3.3892660138e-03,  3.1599117466e-03,  2.9295290953e-03,  2.6982516193e-03,
     2.4662129468e-03,  2.2335466975e-03,  2.0003864075e-03,  1.7668654540e-03,
     1.5331169800e-03,  1.2992738196e-03,  1.0654684241e-03,  8.3183278796e-04,
     5.9849837522e-04,  3.6559604743e-04,  1.3325599097e-04, -9.8392354741e-05,
    -3.2922036707e-04, -5.5910031171e-04, -7.8790541116e-04, -1.0155099138e-03,
    -1.2417891618e-03, -1.4666196584e-03, -1.6898791332e-03, -1.9114466085e-03,
    -2.1312024630e-03, -2.3490284949e-03, -2.5648079850e-03, -2.7784257574e-03,
    -2.9897682395e-03, -3.1987235220e-03, -3.4051814156e-03, -3.6090335086e-03,
    -3.8101732225e-03, -4.0084958659e-03, -4.2038986879e-03, -4.3962809302e-03,
    -4.5855438770e-03, -4.7715909042e-03, -4.9543275285e-03, -5.1336614521e-03,
    -5.3095026091e-03, -5.4817632089e-03, -5.6503577777e-03, -5.8152032003e-03,
    -5.9762187589e-03, -6.1333261709e-03, -6.2864496254e-03, -6.4355158181e-03,
    -6.5804539840e-03, -6.7211959296e-03, -6.8576760631e-03, -6.9898314225e-03,
    -7.1176017030e-03, -7.2409292823e-03, -7.3597592442e-03, -7.4740394008e-03,
    -7.5837203134e-03, -7.6887553106e-03, -7.7891005065e-03, -7.8847148158e-03,
    -7.9755599679e-03, -8.0616005192e-03, -8.1428038641e-03, -8.2191402433e-03,
    -8.2905827524e-03, -8.3571073464e-03, -8.4186928447e-03, -8.4753209335e-03,
    -8.5269761660e-03, -8.5736459624e-03, -8.6153206071e-03, -8.6519932446e-03,
    -8.6836598739e-03, -8.7103193419e-03, -8.7319733335e-03, -8.7486263617e-03,
    -8.7602857561e-03, -8.7669616482e-03, -8.7686669575e-03, -8.7654173744e-03,
    -8.7572313415e-03, -8.7441300350e-03, -8.7261373428e-03, -8.7032798422e-03,
    -8.6755867755e-03, -8.6430900256e-03, -8.6058240873e-03, -8.5638260409e-03,
    -8.5171355209e-03, -8.4657946861e-03, -8.4098481869e-03, -8.3493431315e-03,
    -8.2843290510e-03, -8.2148578635e-03, -8.1409838364e-03, -8.0627635474e-03,
    -7.9802558456e-03, -7.8935218096e-03, -7.8026247058e-03, -7.7076299456e-03,
    -7.6086050402e-03, -7.5056195556e-03, -7.3987450669e-03, -7.2880551099e-03,
    -7.1736251339e-03, -7.0555324516e-03, -6.9338561893e-03, -6.8086772361e-03,
    -6.6800781919e-03, -6.5481433149e-03, -6.4129584680e-03, -6.2746110653e-03,
    -6.1331900164e-03, -5.9887856719e-03, -5.8414897665e-03, -5.6913953627e-03,
    -5.5385967935e-03, -5.3831896048e-03, -5.2252704966e-03, -5.0649372648e-03,
    -4.9022887419e-03, -4.7374247367e-03, -4.5704459759e-03, -4.4014540426e-03,
    -4.2305513160e-03, -4.0578409113e-03, -3.8834266179e-03, -3.7074128386e-03,
    -3.5299045285e-03, -3.3510071334e-03, -3.1708265279e-03, -2.9894689550e-03,
    -2.8070409634e-03, -2.6236493465e-03, -2.4394010813e-03, -2.2544032666e-03,
    -2.0687630619e-03, -1.8825876271e-03, -1.6959840603e-03, -1.5090593388e-03,
    -1.3219202576e-03, -1.1346733702e-03, -9.4742492857e-04, -7.6028082429e-04,
    -5.7334652905e-04, -3.8672703676e-04, -2.0052680570e-04, -1.4849700252e-05,
     1.7020106485e-04,  3.5452298165e-04,  5.3801430470e-04,  7.2057410575e-04,
     9.0210232867e-04,  1.0824998432e-03,  1.2616684978e-03,  1.4395111720e-03,
     1.6159318285e-03,  1.7908355633e-03,  1.9641286560e-03,  2.1357186195e-03,
     2.3055142468e-03,  2.4734256602e-03,  2.6393643563e-03,  2.8032432522e-03,
     2.9649767302e-03,  3.1244806810e-03,  3.2816725465e-03,  3.4364713610e-03,
     3.5887977926e-03,  3.7385741811e-03,  3.8857245782e-03,  4.0301747832e-03,
     4.1718523799e-03,  4.3106867717e-03,  4.4466092149e-03,  4.5795528514e-03,
     4.7094527408e-03,  4.8362458903e-03,  4.9598712836e-03,  5.0802699089e-03,
     5.1973847858e-03,  5.3111609899e-03,  5.4215456782e-03,  5.5284881103e-03,
     5.6319396715e-03,  5.7318538923e-03,  5.8281864673e-03,  5.9208952735e-03,
     6.0099403861e-03,  6.0952840940e-03,  6.1768909132e-03,  6.2547276001e-03,
     6.3287631616e-03,  6.3989688654e-03,  6.4653182491e-03,  6.5277871261e-03,
     6.5863535928e-03,  6.6409980325e-03,  6.6917031186e-03,  6.7384538169e-03,
     6.7812373860e-03,  6.8200433763e-03,  6.8548636288e-03,  6.8856922713e-03,
     6.9125257136e-03,  6.9353626424e-03,  6.9542040137e-03,  6.9690530440e-03,
     6.9799152016e-03,  6.9867981951e-03,  6.9897119613e-03,  6.9886686521e-03,
     6.9836826197e-03,  6.9747704008e-03,  6.9619506999e-03,  6.9452443710e-03,
     6.9246743982e-03,  6.9002658757e-03,  6.8720459856e-03,  6.8400439756e-03,
     6.8042911355e-03,  6.7648207715e-03,  6.7216681812e-03,  6.6748706262e-03,
     6.6244673042e-03,  6.5704993196e-03,  6.5130096546e-03,  6.4520431369e-03,
     6.3876464088e-03,  6.3198678944e-03,  6.2487577650e-03,  6.1743679056e-03,
     6.0967518791e-03,  6.0159648893e-03,  5.9320637450e-03,  5.8451068217e-03,
     5.7551540226e-03,  5.6622667398e-03,  5.5665078139e-03,  5.4679414931e-03,
     5.3666333927e-03,  5.2626504520e-03,  5.1560608919e-03,  5.0469341725e-03,
     4.9353409483e-03,  4.8213530240e-03,  4.7050433107e-03,  4.5864857793e-03,
     4.4657554156e-03,  4.3429281741e-03,  4.2180809313e-03,  4.0912914384e-03,
     3.9626382751e-03,  3.8322008010e-03,  3.7000591084e-03,  3.5662939746e-03,
     3.4309868122e-03,  3.2942196224e-03,  3.1560749451e-03,  3.0166358105e-03,
     2.8759856903e-03,  2.7342084489e-03,  2.5913882939e-03,  2.4476097276e-03,
     2.3029574977e-03,  2.1575165478e-03,  2.0113719696e-03,  1.8646089531e-03,
     1.7173127381e-03,  1.5695685660e-03,  1.4214616308e-03,  1.2730770310e-03,
     1.1244997219e-03,  9.7581446782e-04,  8.2710579401e-04,  6.7845794044e-04,
     5.2995481419e-04,  3.8167994297e-04,  2.3371642977e-04,  8.6146906166e-05,
    -6.0946512369e-05, -2.0748227107e-04, -3.5337942108e-04, -4.9855766252e-04,
    -6.4293738814e-04, -7.8643972606e-04, -9.2898658207e-04, -1.0705006807e-03,
    -1.2109056072e-03, -1.3501258470e-03, -1.4880868258e-03, -1.6247149489e-03,
    -1.7599376389e-03, -1.8936833737e-03, -2.0258817234e-03, -2.1564633864e-03,
    -2.2853602247e-03, -2.4125052986e-03, -2.5378329005e-03, -2.6612785873e-03,
    -2.7827792137e-03, -2.9022729620e-03, -3.0196993737e-03, -3.1349993787e-03,
    -3.2481153232e-03, -3.3589909987e-03, -3.4675716675e-03, -3.5738040895e-03,
    -3.6776365465e-03, -3.7790188669e-03, -3.8779024477e-03, -3.9742402772e-03,
    -4.0679869562e-03, -4.1590987172e-03, -4.2475334443e-03, -4.3332506908e-03,
    -4.4162116962e-03, -4.4963794027e-03, -4.5737184693e-03, -4.6481952865e-03,
    -4.7197779887e-03, -4.7884364667e-03, -4.8541423775e-03, -4.9168691553e-03,
    -4.9765920195e-03, -5.0332879819e-03, -5.0869358548e-03, -5.1375162552e-03,
    -5.1850116098e-03, -5.2294061591e-03, -5.2706859587e-03, -5.3088388817e-03,
    -5.3438546184e-03, -5.3757246761e-03, -5.4044423768e-03, -5.4300028556e-03,
    -5.4524030556e-03, -5.4716417236e-03, -5.4877194053e-03, -5.5006384364e-03,
    -5.5104029371e-03, -5.5170188020e-03, -5.5204936905e-03, -5.5208370164e-03,
    -5.5180599361e-03, -5.5121753357e-03, -5.5031978178e-03, -5.4911436870e-03,
    -5.4760309339e-03, -5.4578792198e-03, -5.4367098586e-03, -5.4125457987e-03,
    -5.3854116051e-03, -5.3553334381e-03, -5.3223390338e-03, -5.2864576822e-03,
    -5.2477202048e-03, -5.2061589316e-03, -5.1618076775e-03, -5.1147017175e-03,
    -5.0648777609e-03, -5.0123739266e-03, -4.9572297145e-03, -4.8994859794e-03,
    -4.8391849021e-03, -4.7763699611e-03, -4.7110859026e-03, -4.6433787111e-03,
    -4.5732955785e-03, -4.5008848723e-03, -4.4261961050e-03, -4.3492799004e-03,
    -4.2701879617e-03, -4.1889730384e-03, -4.1056888913e-03, -4.0203902595e-03,
    -3.9331328251e-03, -3.8439731778e-03, -3.7529687800e-03, -3.6601779309e-03,
    -3.5656597298e-03, -3.4694740401e-03, -3.3716814522e-03, -3.2723432462e-03,
    -3.1715213550e-03, -3.0692783263e-03, -2.9656772848e-03, -2.8607818949e-03,
    -2.7546563212e-03, -2.6473651912e-03, -2.5389735566e-03, -2.4295468546e-03,
    -2.3191508693e-03, -2.2078516933e-03, -2.0957156886e-03, -1.9828094479e-03,
    -1.8691997563e-03, -1.7549535519e-03, -1.6401378879e-03, -1.5248198936e-03,
    -1.4090667360e-03, -1.2929455815e-03, -1.1765235577e-03, -1.0598677154e-03,
    -9.4304499042e-04, -8.2612216669e-04, -7.0916583799e-04, -5.9224237117e-04,
    -4.7541786953e-04, -3.5875813517e-04, -2.4232863390e-04, -1.2619445865e-04,
    -1.0420293429e-05,  1.0492962092e-04,  2.1979152271e-04,  3.3410216337e-04,
     4.4779884162e-04,  5.6081943691e-04,  6.7310244324e-04,  7.8458700098e-04,
     8.9521292975e-04,  1.0049207603e-03,  1.1136517649e-03,  1.2213479889e-03,
     1.3279522802e-03,  1.4334083190e-03,  1.5376606469e-03,  1.6406546951e-03,
     1.7423368123e-03,  1.8426542915e-03,  1.9415553970e-03,  2.0389893895e-03,
     2.1349065521e-03,  2.2292582142e-03,  2.3219967752e-03,  2.4130757288e-03,
     2.5024496834e-03,  2.5900743856e-03,  2.6759067401e-03,  2.7599048302e-03,
     2.8420279374e-03,  2.9222365600e-03,  3.0004924311e-03,  3.0767585354e-03,
     3.1509991262e-03,  3.2231797405e-03,  3.2932672141e-03,  3.3612296957e-03,
     3.4270366596e-03,  3.4906589190e-03,  3.5520686366e-03,  3.6112393362e-03,
     3.6681459120e-03,  3.7227646387e-03,  3.7750731788e-03,  3.8250505905e-03,
     3.8726773351e-03,  3.9179352814e-03,  3.9608077121e-03,  4.0012793273e-03,
     4.0393362480e-03,  4.0749660191e-03,  4.1081576104e-03,  4.1389014179e-03,
     4.1671892640e-03,  4.1930143970e-03,  4.2163714887e-03,  4.2372566333e-03,
     4.2556673434e-03,  4.2716025458e-03,  4.2850625782e-03,  4.2960491822e-03,
     4.3045654977e-03,  4.3106160560e-03,  4.3142067716e-03,  4.3153449335e-03,
     4.3140391966e-03,  4.3102995708e-03,  4.3041374103e-03,  4.2955654030e-03,
     4.2845975569e-03,  4.2712491879e-03,  4.2555369063e-03,  4.2374786014e-03,
     4.2170934279e-03,  4.1944017893e-03,  4.1694253217e-03,  4.1421868770e-03,
     4.1127105051e-03,  4.0810214357e-03,  4.0471460593e-03,  4.0111119086e-03,
     3.9729476371e-03,  3.9326829997e-03,  3.8903488313e-03,  3.8459770245e-03,
     3.7996005083e-03,  3.7512532249e-03,  3.7009701066e-03,  3.6487870524e-03,
     3.5947409036e-03,  3.5388694192e-03,  3.4812112509e-03,  3.4218059182e-03,
     3.3606937817e-03,  3.2979160179e-03,  3.2335145917e-03,  3.1675322302e-03,
     3.1000123952e-03,  3.0309992554e-03,  2.9605376589e-03,  2.8886731051e-03,
     2.8154517159e-03,  2.7409202074e-03,  2.6651258612e-03,  2.5881164945e-03,
     2.5099404314e-03,  2.4306464741e-03,  2.3502838716e-03,  2.2689022914e-03,
     2.1865517893e-03,  2.1032827786e-03,  2.0191460011e-03,  1.9341924962e-03,
     1.8484735710e-03,  1.7620407701e-03,  1.6749458453e-03,  1.5872407247e-03,
     1.4989774835e-03,  1.4102083133e-03,  1.3209854912e-03,  1.2313613513e-03,
     1.1413882532e-03,  1.0511185526e-03,  9.6060457175e-04,  8.6989856921e-04,
     7.7905271079e-04,  6.8811904015e-04,  5.9714944958e-04,  5.0619565046e-04,
     4.1530914573e-04,  3.2454119996e-04,  2.3394281190e-04,  1.4356468668e-04,
     5.3457207111e-05, -3.6329592914e-05, -1.2574605590e-04, -2.1474292778e-04,
    -3.0327138418e-04, -3.9128305632e-04, -4.7873005738e-04, -5.6556500762e-04,
    -6.5174105902e-04, -7.3721192101e-04, -8.2193188348e-04, -9.0585584108e-04,
    -9.8893931695e-04, -1.0711384847e-03, -1.1524101917e-03, -1.2327119809e-03,
    -1.3120021119e-03, -1.3902395821e-03, -1.4673841479e-03, -1.5433963434e-03,
    -1.6182375007e-03, -1.6918697691e-03, -1.7642561325e-03, -1.8353604282e-03,
    -1.9051473637e-03, -1.9735825331e-03, -2.0406324342e-03, -2.1062644832e-03,
    -2.1704470300e-03, -2.2331493726e-03, -2.2943417712e-03, -2.3539954603e-03,
    -2.4120826627e-03, -2.4685766007e-03, -2.5234515071e-03, -2.5766826371e-03,
    -2.6282462771e-03, -2.6781197550e-03, -2.7262814488e-03, -2.7727107946e-03,
    -2.8173882946e-03, -2.8602955237e-03, -2.9014151360e-03, -2.9407308700e-03,
    -2.9782275545e-03, -3.0138911120e-03, -3.0477085626e-03, -3.0796680280e-03,
    -3.1097587327e-03, -3.1379710067e-03, -3.1642962860e-03, -3.1887271137e-03,
    -3.2112571393e-03, -3.2318811187e-03, -3.2505949120e-03, -3.2673954820e-03,
    -3.2822808921e-03, -3.2952503014e-03, -3.3063039626e-03, -3.3154432167e-03,
    -3.3226704878e-03, -3.3279892781e-03, -3.3314041612e-03, -3.3329207749e-03,
    -3.3325458145e-03, -3.3302870245e-03, -3.3261531895e-03, -3.3201541262e-03,
    -3.3123006729e-03, -3.3026046789e-03, -3.2910789951e-03, -3.2777374611e-03,
    -3.2625948945e-03, -3.2456670783e-03, -3.2269707479e-03, -3.2065235777e-03,
    -3.1843441676e-03, -3.1604520285e-03, -3.1348675672e-03, -3.1076120723e-03,
    -3.0787076974e-03, -3.0481774455e-03, -3.0160451526e-03, -2.9823354702e-03,
    -2.9470738489e-03, -2.9102865201e-03, -2.8720004779e-03, -2.8322434608e-03,
    -2.7910439332e-03, -2.7484310657e-03, -2.7044347156e-03, -2.6590854083e-03,
    -2.6124143155e-03, -2.5644532360e-03, -2.5152345751e-03, -2.4647913226e-03,
    -2.4131570330e-03, -2.3603658030e-03, -2.3064522505e-03, -2.2514514927e-03,
    -2.1953991240e-03, -2.1383311934e-03, -2.0802841828e-03, -2.0212949841e-03,
    -1.9614008760e-03, -1.9006395025e-03, -1.8390488484e-03, -1.7766672175e-03,
    -1.7135332088e-03, -1.6496856934e-03, -1.5851637911e-03, -1.5200068478e-03,
    -1.4542544112e-03, -1.3879462079e-03, -1.3211221201e-03, -1.2538221618e-03,
    -1.1860864557e-03, -1.1179552104e-03, -1.0494686958e-03, -9.8066722119e-04,
    -9.1159111155e-04, -8.4228068412e-04, -7.7277622607e-04, -7.0311797134e-04,
    -6.3334607783e-04, -5.6350060485e-04, -4.9362149112e-04, -4.2374853129e-04,
    -3.5392135512e-04, -2.8417940492e-04, -2.1456191343e-04, -1.4510788339e-04,
    -7.5856065182e-05, -6.8449359908e-06,  6.1887320653e-05,  1.3030283592e-04,
     1.9836407592e-04,  2.6603386147e-04,  3.3327538793e-04,  4.0005224544e-04,
     4.6632843667e-04,  5.3206839712e-04,  5.9723701297e-04,  6.6179963884e-04,
     7.2572211666e-04,  7.8897079241e-04,  8.5151253348e-04,  9.1331474565e-04,
     9.7434538943e-04,  1.0345729956e-03,  1.0939666815e-03,  1.1524961660e-03,
     1.2101317838e-03,  1.2668445013e-03,  1.3226059288e-03,  1.3773883350e-03,
     1.4311646606e-03,  1.4839085295e-03,  1.5355942627e-03,  1.5861968890e-03,
     1.6356921569e-03,  1.6840565449e-03,  1.7312672731e-03,  1.7773023116e-03,
     1.8221403907e-03,  1.8657610109e-03,  1.9081444495e-03,  1.9492717703e-03,
     1.9891248303e-03,  2.0276862871e-03,  2.0649396052e-03,  2.1008690627e-03,
     2.1354597563e-03,  2.1686976067e-03,  2.2005693637e-03,  2.2310626089e-03,
     2.2601657612e-03,  2.2878680788e-03,  2.3141596616e-03,  2.3390314548e-03,
     2.3624752491e-03,  2.3844836826e-03,  2.4050502412e-03,  2.4241692593e-03,
     2.4418359185e-03,  2.4580462482e-03,  2.4727971236e-03,  2.4860862633e-03,
     2.4979122288e-03,  2.5082744201e-03,  2.5171730734e-03,  2.5246092580e-03,
     2.5305848708e-03,  2.5351026330e-03,  2.5381660848e-03,  2.5397795796e-03,
     2.5399482782e-03,  2.5386781434e-03,  2.5359759318e-03,  2.5318491876e-03,
     2.5263062351e-03,  2.5193561697e-03,  2.5110088504e-03,  2.5012748911e-03,
     2.4901656502e-03,  2.4776932225e-03,  2.4638704282e-03,  2.4487108028e-03,
     2.4322285864e-03,  2.4144387128e-03,  2.3953567975e-03,  2.3749991267e-03,
     2.3533826448e-03,  2.3305249414e-03,  2.3064442399e-03,  2.2811593830e-03,
     2.2546898200e-03,  2.2270555933e-03,  2.1982773242e-03,  2.1683761987e-03,
     2.1373739531e-03,  2.1052928591e-03,  2.0721557088e-03,  2.0379858000e-03,
     2.0028069200e-03,  1.9666433305e-03,  1.9295197519e-03,  1.8914613460e-03,
     1.8524937014e-03,  1.8126428161e-03,  1.7719350811e-03,  1.7303972637e-03,
     1.6880564908e-03,  1.6449402315e-03,  1.6010762799e-03,  1.5564927388e-03,
     1.5112180008e-03,  1.4652807321e-03,  1.4187098549e-03,  1.3715345284e-03,
     1.3237841330e-03,  1.2754882514e-03,  1.2266766510e-03,  1.1773792664e-03,
     1.1276261816e-03,  1.0774476116e-03,  1.0268738853e-03,  9.7593542731e-04,
     9.2466273960e-04,  8.7308638528e-04,  8.2123696931e-04,  7.6914512146e-04,
     7.1684147899e-04,  6.6435666842e-04,  6.1172128844e-04,  5.5896589280e-04,
     5.0612097268e-04,  4.5321693934e-04,  4.0028410785e-04,  3.4735267927e-04,
     2.9445272408e-04,  2.4161416647e-04,  1.8886676631e-04,  1.3624010420e-04,
     8.3763564741e-05,  3.1466320309e-05, -2.0622684252e-05, -7.2474747267e-05,
    -1.2406142581e-04, -1.7535455085e-04, -2.2632624146e-04, -2.7694892099e-04,
    -3.2719533012e-04, -3.7703854163e-04, -4.2645197487e-04, -4.7540940828e-04,
    -5.2388499381e-04, -5.7185326998e-04, -6.1928917434e-04, -6.6616805643e-04,
    -7.1246569022e-04, -7.5815828566e-04, -8.0322250046e-04, -8.4763545244e-04,
    -8.9137472902e-04, -9.3441839951e-04, -9.7674502498e-04, -1.0183336678e-03,
    -1.0591639026e-03, -1.0992158249e-03, -1.1384700602e-03, -1.1769077736e-03,
    -1.2145106775e-03, -1.2512610396e-03, -1.2871416911e-03, -1.3221360342e-03,
    -1.3562280480e-03, -1.3894022967e-03, -1.4216439350e-03, -1.4529387137e-03,
    -1.4832729865e-03, -1.5126337137e-03, -1.5410084680e-03, -1.5683854387e-03,
    -1.5947534359e-03, -1.6201018936e-03, -1.6444208741e-03, -1.6677010697e-03,
    -1.6899338060e-03, -1.7111110446e-03, -1.7312253833e-03, -1.7502700593e-03,
    -1.7682389495e-03, -1.7851265710e-03, -1.8009280817e-03, -1.8156392803e-03,
    -1.8292566059e-03, -1.8417771369e-03, -1.8531985910e-03, -1.8635193217e-03,
    -1.8727383183e-03, -1.8808552032e-03, -1.8878702281e-03, -1.8937842734e-03,
    -1.8985988429e-03, -1.9023160612e-03, -1.9049386694e-03, -1.9064700213e-03,
    -1.9069140777e-03, -1.9062754027e-03, -1.9045591583e-03, -1.9017710974e-03,
    -1.8979175599e-03, -1.8930054651e-03, -1.8870423053e-03, -1.8800361400e-03,
    -1.8719955874e-03, -1.8629298180e-03, -1.8528485466e-03, -1.8417620243e-03,
    -1.8296810299e-03, -1.8166168623e-03, -1.8025813309e-03, -1.7875867466e-03,
    -1.7716459139e-03, -1.7547721188e-03, -1.7369791216e-03, -1.7182811455e-03,
    -1.6986928667e-03, -1.6782294043e-03, -1.6569063093e-03, -1.6347395539e-03,
    -1.6117455204e-03, -1.5879409904e-03, -1.5633431327e-03, -1.5379694924e-03,
    -1.5118379793e-03, -1.4849668546e-03, -1.4573747212e-03, -1.4290805094e-03,
    -1.4001034656e-03, -1.3704631398e-03, -1.3401793734e-03, -1.3092722853e-03,
    -1.2777622606e-03, -1.2456699371e-03, -1.2130161915e-03, -1.1798221283e-03,
    -1.1461090651e-03, -1.1118985198e-03, -1.0772121980e-03, -1.0420719789e-03,
    -1.0064999022e-03, -9.7051815544e-04, -9.3414905964e-04, -8.9741505643e-04,
    -8.6033869498e-04, -8.2294261786e-04, -7.8524954777e-04, -7.4728227517e-04,
    -7.0906364328e-04, -6.7061653618e-04, -6.3196386511e-04, -5.9312855451e-04,
    -5.5413352983e-04, -5.1500170389e-04, -4.7575596367e-04, -4.3641915772e-04,
    -3.9701408329e-04, -3.5756347282e-04, -3.1808998183e-04, -2.7861617646e-04,
    -2.3916451970e-04, -1.9975736092e-04, -1.6041692201e-04, -1.2116528576e-04,
    -8.2024384107e-05, -4.3015985781e-05, -4.1616846182e-06,  3.4517111558e-05,
     7.2999192630e-05,  1.1126355790e-04,  1.4928942609e-04,  1.8705624735e-04,
     2.2454371381e-04,  2.6173176903e-04,  2.9860061997e-04,  3.3513074590e-04,
     3.7130290866e-04,  4.0709816285e-04,  4.4249786497e-04,  4.7748368256e-04,
     5.1203760367e-04,  5.4614194574e-04,  5.7977936328e-04,  6.1293285799e-04,
     6.4558578542e-04,  6.7772186339e-04,  7.0932518024e-04,  7.4038020112e-04,
     7.7087177592e-04,  8.0078514626e-04,  8.3010595176e-04,  8.5882023657e-04,
     8.8691445590e-04,  9.1437548137e-04,  9.4119060641e-04,  9.6734755292e-04,
     9.9283447446e-04,  1.0176399627e-03,  1.0417530513e-03,  1.0651632197e-03,
     1.0878603978e-03,  1.1098349696e-03,  1.1310777761e-03,  1.1515801187e-03,
     1.1713337628e-03,  1.1903309389e-03,  1.2085643462e-03,  1.2260271544e-03,
     1.2427130042e-03,  1.2586160108e-03,  1.2737307637e-03,  1.2880523277e-03,
     1.3015762443e-03,  1.3142985311e-03,  1.3262156823e-03,  1.3373246688e-03,
     1.3476229374e-03,  1.3571084098e-03,  1.3657794824e-03,  1.3736350241e-03,
     1.3806743751e-03,  1.3868973457e-03,  1.3923042129e-03,  1.3968957194e-03,
     1.4006730705e-03,  1.4036379309e-03,  1.4057924220e-03,  1.4071391185e-03,
     1.4076810449e-03,  1.4074216710e-03,  1.4063649095e-03,  1.4045151094e-03,
     1.4018770533e-03,  1.3984559520e-03,  1.3942574388e-03,  1.3892875657e-03,
     1.3835527970e-03,  1.3770600040e-03,  1.3698164588e-03,  1.3618298289e-03,
     1.3531081699e-03,  1.3436599204e-03,  1.3334938945e-03,  1.3226192747e-03,
     1.3110456061e-03,  1.2987827881e-03,  1.2858410671e-03,  1.2722310299e-03,
     1.2579635951e-03,  1.2430500058e-03,  1.2275018213e-03,  1.2113309095e-03,
     1.1945494374e-03,  1.1771698644e-03,  1.1592049327e-03,  1.1406676584e-03,
     1.1215713242e-03,  1.1019294686e-03,  1.0817558784e-03,  1.0610645788e-03,
     1.0398698246e-03,  1.0181860908e-03,  9.9602806338e-04,  9.7341062977e-04,
     9.5034886901e-04,  9.2685804314e-04,  9.0295358622e-04,  8.7865109578e-04,
     8.5396632309e-04,  8.2891516210e-04,  8.0351364139e-04,  7.7777791298e-04,
     7.5172424294e-04,  7.2536900169e-04,  6.9872865413e-04,  6.7181974913e-04,
     6.4465891037e-04,  6.1726282621e-04,  5.8964823902e-04,  5.6183193681e-04,
     5.3383074225e-04,  5.0566150281e-04,  4.7734108213e-04,  4.4888634864e-04,
     4.2031416694e-04,  3.9164138788e-04,  3.6288483893e-04,  3.3406131433e-04,
     3.0518756644e-04,  2.7628029540e-04,  2.4735613982e-04,  2.1843166870e-04,
     1.8952337045e-04,  1.6064764531e-04,  1.3182079596e-04,  1.0305901794e-04,
     7.4378391705e-05,  4.5794873620e-05,  1.7324287146e-05, -1.1017685271e-05,
    -3.9215510220e-05, -6.7253812496e-05, -9.5117382422e-05, -1.2279118354e-04,
    -1.5026036191e-04, -1.7751025159e-04, -2.0452638380e-04, -2.3129449377e-04,
    -2.5780052767e-04, -2.8403065003e-04, -3.0997125085e-04, -3.3560895164e-04,
    -3.6093061226e-04, -3.8592333797e-04, -4.1057448433e-04, -4.3487166439e-04,
    -4.5880275462e-04, -4.8235589924e-04, -5.0551951750e-04, -5.2828230771e-04,
    -5.5063325270e-04, -5.7256162515e-04, -5.9405699218e-04, -6.1510921948e-04,
    -6.3570847623e-04, -6.5584523934e-04, -6.7551029620e-04, -6.9469475045e-04,
    -7.1339002400e-04, -7.3158786075e-04, -7.4928033057e-04, -7.6645983103e-04,
    -7.8311909082e-04, -7.9925117247e-04, -8.1484947439e-04, -8.2990773292e-04,
    -8.4442002494e-04, -8.5838076890e-04, -8.7178472613e-04, -8.8462700360e-04,
    -8.9690305305e-04, -9.0860867349e-04, -9.1974001157e-04, -9.3029356132e-04,
    -9.4026616535e-04, -9.4965501455e-04, -9.5845764794e-04, -9.6667195237e-04,
    -9.7429616279e-04, -9.8132886010e-04, -9.8776897145e-04, -9.9361576900e-04,
    -9.9886886755e-04, -1.0035282247e-03, -1.0075941383e-03, -1.0110672442e-03,
    -1.0139485154e-03, -1.0162392587e-03, -1.0179411125e-03, -1.0190560448e-03,
    -1.0195863503e-03, -1.0195346464e-03, -1.0189038719e-03, -1.0176972823e-03,
    -1.0159184465e-03, -1.0135712440e-03, -1.0106598603e-03, -1.0071887833e-03,
    -1.0031627996e-03, -9.9858698940e-04, -9.9346672261e-04, -9.8780765453e-04,
    -9.8161572081e-04, -9.7489713238e-04, -9.6765837187e-04, -9.5990618650e-04,
    -9.5164758437e-04, -9.4288982889e-04, -9.3364043238e-04, -9.2390715186e-04,
    -9.1369798262e-04, -9.0302115239e-04, -8.9188511557e-04, -8.8029854759e-04,
    -8.6827033779e-04, -8.5580958431e-04, -8.4292558776e-04, -8.2962784361e-04,
    -8.1592603752e-04, -8.0183003753e-04, -7.8734988744e-04, -7.7249580127e-04,
    -7.5727815535e-04, -7.4170748209e-04, -7.2579446313e-04, -7.0954992251e-04,
    -6.9298481879e-04, -6.7611023969e-04, -6.5893739377e-04, -6.4147760348e-04,
    -6.2374229932e-04, -6.0574301083e-04, -5.8749136069e-04, -5.6899905745e-04,
    -5.5027788777e-04, -5.3133970976e-04, -5.1219644595e-04, -4.9286007553e-04,
    -4.7334262719e-04, -4.5365617287e-04, -4.3381281907e-04, -4.1382470087e-04,
    -3.9370397501e-04, -3.7346281121e-04, -3.5311338703e-04, -3.3266787939e-04,
    -3.1213845808e-04, -2.9153727876e-04, -2.7087647648e-04, -2.5016815772e-04,
    -2.2942439465e-04, -2.0865721803e-04, -1.8787860954e-04, -1.6710049680e-04,
    -1.4633474568e-04, -1.2559315357e-04, -1.0488744432e-04, -8.4229260112e-05,
    -6.3630156232e-05, -4.3101594603e-05, -2.2654937848e-05, -2.3014425533e-06,
     1.7947745394e-05,  3.8081597810e-05,  5.8089210022e-05,  7.7959805110e-05,
     9.7682741408e-05,  1.1724751650e-04,  1.3664377270e-04,  1.5586130315e-04,
     1.7489005610e-04,  1.9372014013e-04,  2.1234182925e-04,  2.3074556760e-04,
     2.4892197325e-04,  2.6686184416e-04,  2.8455616156e-04,  3.0199609396e-04,
     3.1917300287e-04,  3.3607844463e-04,  3.5270417572e-04,  3.6904215616e-04,
     3.8508455272e-04,  4.0082374267e-04,  4.1625231747e-04,  4.3136308522e-04,
     4.4614907367e-04,  4.6060353421e-04,  4.7471994286e-04,  4.8849200422e-04,
     5.0191365396e-04,  5.1497905976e-04,  5.2768262511e-04,  5.4001899024e-04,
     5.5198303409e-04,  5.6356987642e-04,  5.7477487937e-04,  5.8559364831e-04,
     5.9602203359e-04,  6.0605613203e-04,  6.1569228616e-04,  6.2492708750e-04,
     6.3375737538e-04,  6.4218023795e-04,  6.5019301330e-04,  6.5779328861e-04,
     6.6497890056e-04,  6.7174793573e-04,  6.7809872995e-04,  6.8402986779e-04,
     6.8954018290e-04,  6.9462875667e-04,  6.9929491691e-04,  7.0353823882e-04,
     7.0735854158e-04,  7.1075588895e-04,  7.1373058754e-04,  7.1628318466e-04,
     7.1841446745e-04,  7.2012546101e-04,  7.2141742626e-04,  7.2229185800e-04,
     7.2275048380e-04,  7.2279526010e-04,  7.2242837106e-04,  7.2165222622e-04,
     7.2046945635e-04,  7.1888291285e-04,  7.1689566370e-04,  7.1451099051e-04,
     7.1173238622e-04,  7.0856355135e-04,  7.0500839068e-04,  7.0107101024e-04,
     6.9675571433e-04,  6.9206700039e-04,  6.8700955758e-04,  6.8158826136e-04,
     6.7580816979e-04,  6.6967452088e-04,  6.6319272717e-04,  6.5636837241e-04,
     6.4920720766e-04,  6.4171514648e-04,  6.3389826063e-04,  6.2576277638e-04,
     6.1731506951e-04,  6.0856166052e-04,  5.9950921170e-04,  5.9016452035e-04,
     5.8053451546e-04,  5.7062625317e-04,  5.6044691062e-04,  5.5000378278e-04,
     5.3930427680e-04,  5.2835590720e-04,  5.1716629071e-04,  5.0574314230e-04,
     4.9409426852e-04,  4.8222756395e-04,  4.7015100607e-04,  4.5787264891e-04,
     4.4540061976e-04,  4.3274311293e-04,  4.1990838437e-04,  4.0690474787e-04,
     3.9374056875e-04,  3.8042425934e-04,  3.6696427364e-04,  3.5336910278e-04,
     3.3964726830e-04,  3.2580731928e-04,  3.1185782561e-04,  2.9780737315e-04,
     2.8366455997e-04,  2.6943798948e-04,  2.5513626642e-04,  2.4076799206e-04,
     2.2634175852e-04,  2.1186614429e-04,  1.9734970968e-04,  1.8280099146e-04,
     1.6822849768e-04,  1.5364070439e-04,  1.3904604891e-04,  1.2445292656e-04,
     1.0986968624e-04,  9.5304624323e-05,  8.0765981890e-05,  6.6261939179e-05,
     5.1800611318e-05,  3.7390044160e-05,  2.3038210511e-05,  8.7530047847e-06,
    -5.4577601933e-06, -1.9586357133e-05, -3.3625149860e-05, -4.7566595062e-05,
    -6.1403247458e-05, -7.5127763719e-05, -8.8732904839e-05, -1.0221154121e-04,
    -1.1555665558e-04, -1.2876134633e-04, -1.4181883079e-04, -1.5472244930e-04,
    -1.6746566695e-04, -1.8004207773e-04, -1.9244540802e-04, -2.0466951749e-04,
    -2.1670840439e-04, -2.2855620650e-04, -2.4020720405e-04, -2.5165582286e-04,
    -2.6289663637e-04, -2.7392436768e-04, -2.8473389243e-04, -2.9532024073e-04,
    -3.0567859821e-04, -3.1580430945e-04, -3.2569287905e-04, -3.3533997254e-04,
    -3.4474142031e-04, -3.5389321637e-04, -3.6279152153e-04, -3.7143266413e-04,
    -3.7981314125e-04, -3.8792961948e-04, -3.9577893697e-04, -4.0335810319e-04,
    -4.1066429974e-04, -4.1769488225e-04, -4.2444737913e-04, -4.3091949317e-04,
    -4.3710910244e-04, -4.4301425869e-04, -4.4863318945e-04, -4.5396429690e-04,
    -4.5900615800e-04, -4.6375752434e-04, -4.6821732264e-04, -4.7238465307e-04,
    -4.7625878973e-04, -4.7983918041e-04, -4.8312544406e-04, -4.8611737247e-04,
    -4.8881492793e-04, -4.9121824226e-04, -4.9332761696e-04, -4.9514352080e-04,
    -4.9666658899e-04, -4.9789762209e-04, -4.9883758472e-04, -4.9948760314e-04,
    -4.9984896527e-04, -4.9992311786e-04, -4.9971166435e-04, -4.9921636488e-04,
    -4.9843913217e-04, -4.9738203094e-04, -4.9604727600e-04, -4.9443722903e-04,
    -4.9255439719e-04, -4.9040143070e-04, -4.8798112030e-04, -4.8529639445e-04,
    -4.8235031850e-04, -4.7914608985e-04, -4.7568703692e-04, -4.7197661643e-04,
    -4.6801840920e-04, -4.6381611914e-04, -4.5937356967e-04, -4.5469470034e-04,
    -4.4978356489e-04, -4.4464432756e-04, -4.3928125994e-04, -4.3369873796e-04,
    -4.2790123987e-04, -4.2189334058e-04, -4.1567971150e-04, -4.0926511502e-04,
    -4.0265440165e-04, -3.9585250762e-04, -3.8886445050e-04, -3.8169532607e-04,
    -3.7435030563e-04, -3.6683463173e-04, -3.5915361467e-04, -3.5131262971e-04,
    -3.4331711326e-04, -3.3517255837e-04, -3.2688451378e-04, -3.1845857742e-04,
    -3.0990039469e-04, -3.0121565471e-04, -2.9241008579e-04, -2.8348945291e-04,
    -2.7445955404e-04, -2.6532621616e-04, -2.5609529166e-04, -2.4677265582e-04,
    -2.3736420153e-04, -2.2787583696e-04, -2.1831348259e-04, -2.0868306563e-04,
    -1.9899051896e-04, -1.8924177616e-04, -1.7944276796e-04, -1.6959941975e-04,
    -1.5971764759e-04, -1.4980335472e-04, -1.3986242857e-04, -1.2990073779e-04,
    -1.1992412724e-04, -1.0993841696e-04, -9.9949397617e-05, -8.9962826915e-05,
    -7.9984428447e-05, -7.0019886250e-05, -6.0074842988e-05, -5.0154896790e-05,
    -4.0265598031e-05, -3.0412446111e-05, -2.0600887678e-05, -1.0836312746e-05,
    -1.1240516679e-06,  8.5306260705e-06,  1.8122516553e-05,  2.7646483029e-05,
     3.7097457591e-05,  4.6470445978e-05,  5.5760527745e-05,  6.4962860155e-05,
     7.4072680009e-05,  8.3085306462e-05,  9.1996141801e-05,  1.0080067566e-04,
     1.0949448556e-04,  1.1807323909e-04,  1.2653269742e-04,  1.3486871525e-04,
     1.4307724350e-04,  1.5115433186e-04,  1.5909612860e-04,  1.6689888405e-04,
     1.7455895136e-04,  1.8207278831e-04,  1.8943695792e-04,  1.9664813160e-04,
     2.0370308806e-04,  2.1059871618e-04,  2.1733201645e-04,  2.2390010016e-04,
     2.3030019265e-04,  2.3652963300e-04,  2.4258587451e-04,  2.4846648662e-04,
     2.5416915503e-04,  2.5969168214e-04,  2.6503198811e-04,  2.7018811170e-04,
     2.7515820890e-04,  2.7994055557e-04,  2.8453354650e-04,  2.8893569501e-04,
     2.9314563533e-04,  2.9716212026e-04,  3.0098402235e-04,  3.0461033387e-04,
     3.0804016627e-04,  3.1127274985e-04,  3.1430743444e-04,  3.1714368832e-04,
     3.1978109710e-04,  3.2221936545e-04,  3.2445831381e-04,  3.2649787950e-04,
     3.2833811638e-04,  3.2997919248e-04,  3.3142139069e-04,  3.3266510744e-04,
     3.3371085122e-04,  3.3455924205e-04,  3.3521101103e-04,  3.3566699814e-04,
     3.3592815151e-04,  3.3599552744e-04,  3.3587028611e-04,  3.3555369363e-04,
     3.3504711859e-04,  3.3435203099e-04,  3.3347000150e-04,  3.3240269940e-04,
     3.3115189045e-04,  3.2971943622e-04,  3.2810729216e-04,  3.2631750507e-04,
     3.2435221276e-04,  3.2221364166e-04,  3.1990410359e-04,  3.1742599680e-04,
     3.1478180088e-04,  3.1197407688e-04,  3.0900546494e-04,  3.0587868188e-04,
     3.0259651903e-04,  2.9916184095e-04,  2.9557758242e-04,  2.9184674612e-04,
     2.8797240223e-04,  2.8395768397e-04,  2.7980578653e-04,  2.7551996555e-04,
     2.7110353255e-04,  2.6655985499e-04,  2.6189235293e-04,  2.5710449653e-04,
     2.5219980422e-04,  2.4718184044e-04,  2.4205421235e-04,  2.3682056803e-04,
     2.3148459507e-04,  2.2605001573e-04,  2.2052058744e-04,  2.1490009865e-04,
     2.0919236604e-04,  2.0340123381e-04,  1.9753056967e-04,  1.9158426308e-04,
     1.8556622312e-04,  1.7948037603e-04,  1.7333066172e-04,  1.6712103305e-04,
     1.6085545240e-04,  1.5453788866e-04,  1.4817231729e-04,  1.4176271530e-04,
     1.3531306011e-04,  1.2882732788e-04,  1.2230948936e-04,  1.1576350924e-04,
     1.0919334330e-04,  1.0260293639e-04,  9.5996219170e-05,  8.9377108135e-05,
     8.2749500508e-05,  7.6117274032e-05,  6.9484285038e-05,  6.2854364520e-05,
     5.6231318031e-05,  4.9618922894e-05,  4.3020925141e-05,  3.6441039183e-05,
     2.9882944837e-05,  2.3350285625e-05,  1.6846666687e-05,  1.0375654283e-05,
     3.9407709467e-06, -2.4545029361e-06, -8.8067329427e-06, -1.5112532384e-05,
    -2.1368562197e-05, -2.7571534424e-05, -3.3718213322e-05, -3.9805416301e-05,
    -4.5830016186e-05, -5.1788942772e-05, -5.7679183265e-05, -6.3497784196e-05,
    -6.9241853969e-05, -7.4908561415e-05, -8.0495140147e-05, -8.5998888071e-05,
    -9.1417167945e-05, -9.6747410415e-05, -1.0198711337e-04, -1.0713384367e-04,
    -1.1218523829e-04, -1.1713900550e-04, -1.2199292422e-04, -1.2674484690e-04,
    -1.3139269913e-04, -1.3593447959e-04, -1.4036826345e-04, -1.4469220014e-04,
    -1.4890451527e-04, -1.5300351167e-04, -1.5698756837e-04, -1.6085514210e-04,
    -1.6460476788e-04, -1.6823505891e-04, -1.7174470638e-04, -1.7513248147e-04,
    -1.7839723329e-04, -1.8153789008e-04, -1.8455346064e-04, -1.8744303139e-04,
    -1.9020576955e-04, -1.9284092126e-04, -1.9534781161e-04, -1.9772584489e-04,
    -1.9997450482e-04, -2.0209335315e-04, -2.0408203058e-04, -2.0594025652e-04,
    -2.0766782704e-04, -2.0926461644e-04, -2.1073057622e-04, -2.1206573307e-04,
    -2.1327019159e-04, -2.1434413054e-04, -2.1528780398e-04, -2.1610154002e-04,
    -2.1678574064e-04, -2.1734087960e-04, -2.1776750393e-04, -2.1806623153e-04,
    -2.1823774998e-04, -2.1828281774e-04, -2.1820226079e-04, -2.1799697265e-04,
    -2.1766791479e-04, -2.1721611305e-04, -2.1664265879e-04, -2.1594870681e-04,
    -2.1513547432e-04, -2.1420423930e-04, -2.1315634111e-04, -2.1199317712e-04,
    -2.1071620267e-04, -2.0932693078e-04, -2.0782692795e-04, -2.0621781588e-04,
    -2.0450126880e-04, -2.0267901162e-04, -2.0075282008e-04, -1.9872451840e-04,
    -1.9659597763e-04, -1.9436911463e-04, -1.9204589133e-04, -1.8962831118e-04,
    -1.8711842038e-04, -1.8451830503e-04, -1.8183008851e-04, -1.7905593277e-04,
    -1.7619803400e-04, -1.7325862237e-04, -1.7023996109e-04, -1.6714434382e-04,
    -1.6397409334e-04, -1.6073156044e-04, -1.5741912217e-04, -1.5403917882e-04,
    -1.5059415568e-04, -1.4708649785e-04, -1.4351867074e-04, -1.3989315871e-04,
    -1.3621246153e-04, -1.3247909448e-04, -1.2869558673e-04, -1.2486447867e-04,
    -1.2098832136e-04, -1.1706967496e-04, -1.1311110633e-04, -1.0911518740e-04,
    -1.0508449585e-04, -1.0102160957e-04, -9.6929109405e-05, -9.2809575053e-05,
    -8.8665583629e-05, -8.4499709279e-05, -8.0314520972e-05, -7.6112580587e-05,
    -7.1896442656e-05, -6.7668652465e-05, -6.3431743560e-05, -5.9188237820e-05,
    -5.4940643700e-05, -5.0691452928e-05, -4.6443143046e-05, -4.2198172267e-05,
    -3.7958980084e-05, -3.3727986033e-05, -2.9507587299e-05, -2.5300158002e-05,
    -2.1108048569e-05, -1.6933583969e-05, -1.2779061749e-05, -8.6467531740e-06,
    -4.5388992208e-06, -4.5771077505e-07,  3.5946311355e-06,  7.6159794021e-06,
     1.1604218952e-05,  1.5557269073e-05,  1.9473084947e-05,  2.3349657761e-05,
     2.7185015534e-05,  3.0977224886e-05,  3.4724391359e-05,  3.8424659042e-05,
     4.2076214553e-05,  4.5677284562e-05,  4.9226138010e-05,  5.2721087922e-05,
     5.6160489179e-05,  5.9542741834e-05,  6.2866290692e-05,  6.6129625953e-05,
     6.9331283543e-05,  7.2469847095e-05,  7.5543946099e-05,  7.8552257741e-05,
     8.1493508276e-05,  8.4366470716e-05,  8.7169968110e-05,  8.9902872945e-05,
     9.2564105725e-05,  9.5152638223e-05,  9.7667491240e-05,  1.0010773608e-04,
     1.0247249424e-04,  1.0476093901e-04,  1.0697229253e-04,  1.0910582933e-04,
     1.1116087430e-04,  1.1313680225e-04,  1.1503304057e-04,  1.1684906667e-04,
     1.1858440854e-04,  1.2023864604e-04,  1.2181140851e-04,  1.2330237608e-04,
     1.2471127938e-04,  1.2603789957e-04,  1.2728206638e-04,  1.2844366126e-04,
     1.2952261325e-04,  1.3051890057e-04,  1.3143255142e-04,  1.3226364062e-04,
     1.3301229212e-04,  1.3367867743e-04,  1.3426301445e-04,  1.3476556783e-04,
     1.3518664883e-04,  1.3552661361e-04,  1.3578586342e-04,  1.3596484537e-04,
     1.3606404842e-04,  1.3608400615e-04,  1.3602529482e-04,  1.3588853127e-04,
     1.3567437549e-04,  1.3538352724e-04,  1.3501672622e-04,  1.3457475144e-04,
     1.3405842073e-04,  1.3346858841e-04,  1.3280614684e-04,  1.3207202467e-04,
     1.3126718433e-04,  1.3039262487e-04,  1.2944937729e-04,  1.2843850543e-04,
     1.2736110603e-04,  1.2621830587e-04,  1.2501126220e-04,  1.2374116184e-04,
     1.2240921955e-04,  1.2101667672e-04,  1.1956480249e-04,  1.1805489052e-04,
     1.1648825868e-04,  1.1486625035e-04,  1.1319022886e-04,  1.1146158087e-04,
     1.0968171311e-04,  1.0785205158e-04,  1.0597404128e-04,  1.0404914536e-04,
     1.0207884270e-04,  1.0006462804e-04,  9.8008011468e-05,  9.5910515158e-05,
     9.3773675049e-05,  9.1599039209e-05,  8.9388164368e-05,  8.7142619126e-05,
     8.4863979018e-05,  8.2553827450e-05,  8.0213754755e-05,  7.7845357071e-05,
     7.5450234310e-05,  7.3029990912e-05,  7.0586233853e-05,  6.8120570241e-05,
     6.5634610103e-05,  6.3129961696e-05,  6.0608232118e-05,  5.8071027665e-05,
     5.5519949790e-05,  5.2956596559e-05,  5.0382561297e-05,  4.7799431087e-05,
     4.5208786033e-05,  4.2612199662e-05,  4.0011235977e-05,  3.7407449042e-05,
     3.4802384587e-05,  3.2197574606e-05,  2.9594541113e-05,  2.6994793179e-05,
     2.4399825308e-05,  2.1811118542e-05,  1.9230138559e-05,  1.6658334627e-05,
     1.4097139687e-05,  1.1547970285e-05,  9.0122232304e-06,  6.4912777663e-06,
     3.9864938426e-06,  1.4992093827e-06, -9.6925600492e-07, -3.4176054407e-06,
    -5.8445640543e-06, -8.2488786137e-06, -1.0629319914e-05, -1.2984682608e-05,
    -1.5313785194e-05, -1.7615470449e-05, -1.9888608087e-05, -2.2132091046e-05,
    -2.4344840469e-05, -2.6525804074e-05, -2.8673955273e-05, -3.0788296645e-05,
    -3.2867857702e-05, -3.4911695691e-05, -3.6918897431e-05, -3.8888578380e-05,
    -4.0819882795e-05, -4.2711984866e-05, -4.4564089070e-05, -4.6375427623e-05,
    -4.8145265822e-05, -4.9872897489e-05, -5.1557647345e-05, -5.3198872177e-05,
    -5.4795958094e-05, -5.6348322743e-05, -5.7855415230e-05, -5.9316715619e-05,
    -6.0731734831e-05, -6.2100016584e-05, -6.3421134747e-05, -6.4694694169e-05,
    -6.5920332937e-05, -6.7097717989e-05, -6.8226549238e-05, -6.9306558094e-05,
    -7.0337505652e-05, -7.1319185350e-05, -7.2251420923e-05, -7.3134066746e-05,
    -7.3967007607e-05, -7.4750160337e-05, -7.5483469750e-05, -7.6166912393e-05,
    -7.6800494495e-05, -7.7384250050e-05, -7.7918244723e-05, -7.8402572034e-05,
    -7.8837354073e-05, -7.9222742486e-05, -7.9558916173e-05, -7.9846081556e-05,
    -8.0084473067e-05, -8.0274352648e-05, -8.0416007310e-05, -8.0509752923e-05,
    -8.0555929494e-05, -8.0554902160e-05, -8.0507063173e-05, -8.0412827062e-05,
    -8.0272634128e-05, -8.0086948372e-05, -7.9856256706e-05, -7.9581068301e-05,
    -7.9261915930e-05, -7.8899352834e-05, -7.8493953716e-05, -7.8046315827e-05,
    -7.7557054282e-05, -7.7026805187e-05, -7.6456224337e-05, -7.5845983915e-05,
    -7.5196776589e-05, -7.4509311404e-05, -7.3784314572e-05, -7.3022528338e-05,
    -7.2224711884e-05, -7.1391637335e-05, -7.0524093207e-05, -6.9622882176e-05,
    -6.8688818451e-05, -6.7722731611e-05, -6.6725461804e-05, -6.5697860266e-05,
    -6.4640790833e-05, -6.3555126423e-05, -6.2441750112e-05, -6.1301554543e-05,
    -6.0135440957e-05, -5.8944316859e-05, -5.7729099321e-05, -5.6490710627e-05,
    -5.5230078745e-05, -5.3948139655e-05, -5.2645831178e-05, -5.1324097037e-05,
    -4.9983884414e-05, -4.8626142807e-05, -4.7251824498e-05, -4.5861884526e-05,
    -4.4457278158e-05, -4.3038961104e-05, -4.1607890817e-05, -4.0165021644e-05,
    -3.8711308545e-05, -3.7247705532e-05, -3.5775162091e-05, -3.4294627479e-05,
    -3.2807046129e-05, -3.1313358448e-05, -2.9814500967e-05, -2.8311405748e-05,
    -2.6804998254e-05, -2.5296198977e-05, -2.3785922435e-05, -2.2275073465e-05,
    -2.0764552764e-05, -1.9255250978e-05, -1.7748050524e-05, -1.6243826513e-05,
    -1.4743443094e-05, -1.3247754791e-05, -1.1757606636e-05, -1.0273832441e-05,
    -8.7972545674e-06, -7.3286854427e-06, -5.8689247276e-06, -4.4187585519e-06,
    -2.9789635555e-06, -1.5502995596e-06, -1.3351574843e-07,  1.2706526834e-06,
     2.6614854151e-06,  4.0382759202e-06,  5.4003319185e-06,  6.7469772257e-06,
     8.0775504331e-06,  9.3914041685e-06,  1.0687908949e-05,  1.1966449967e-05,
     1.3226427878e-05,  1.4467262871e-05,  1.5688388352e-05,  1.6889256404e-05,
     1.8069336025e-05,  1.9228112530e-05,  2.0365089055e-05,  2.1479787202e-05,
     2.2571745344e-05,  2.3640519043e-05,  2.4685684116e-05,  2.5706830983e-05,
     2.6703570458e-05,  2.7675531988e-05,  2.8622361050e-05,  2.9543723858e-05,
     3.0439303782e-05,  3.1308801939e-05,  3.2151938700e-05,  3.2968453258e-05,
     3.3758101977e-05,  3.4520660669e-05,  3.5255924261e-05,  3.5963702805e-05,
     3.6643828359e-05,  3.7296149060e-05,  3.7920531048e-05,  3.8516860729e-05,
     3.9085040356e-05,  3.9624990596e-05,  4.0136650171e-05,  4.0619975608e-05,
     4.1074939493e-05,  4.1501534614e-05,  4.1899769023e-05,  4.2269667296e-05,
     4.2611273480e-05,  4.2924645168e-05,  4.3209858183e-05,  4.3467005736e-05,
     4.3696194971e-05,  4.3897550682e-05,  4.4071212702e-05,  4.4217336275e-05,
     4.4336090909e-05,  4.4427664265e-05,  4.4492254989e-05,  4.4530078862e-05,
     4.4541365774e-05,  4.4526357091e-05,  4.4485310640e-05,  4.4418496703e-05,
     4.4326197916e-05,  4.4208711448e-05,  4.4066346065e-05,  4.3899422113e-05,
     4.3708272358e-05,  4.3493242409e-05,  4.3254686068e-05,  4.2992972209e-05,
     4.2708477794e-05,  4.2401589466e-05,  4.2072706450e-05,  4.1722234778e-05,
     4.1350591280e-05,  4.0958202151e-05,  4.0545501813e-05,  4.0112931836e-05,
     3.9660943730e-05,  3.9189994847e-05,  3.8700549342e-05,  3.8193081417e-05,
     3.7668068149e-05,  3.7125994594e-05,  3.6567352371e-05,  3.5992635558e-05,
     3.5402346093e-05,  3.4796989732e-05,  3.4177076653e-05,  3.3543120504e-05,
     3.2895640643e-05,  3.2235156255e-05,  3.1562191751e-05,  3.0877274905e-05,
     3.0180932574e-05,  2.9473697466e-05,  2.8756101610e-05,  2.8028677452e-05,
     2.7291960236e-05,  2.6546484524e-05,  2.5792784957e-05,  2.5031397040e-05,
     2.4262856130e-05,  2.3487693968e-05,  2.2706444615e-05,  2.1919638697e-05,
     2.1127804002e-05,  2.0331470412e-05,  1.9531161072e-05,  1.8727398642e-05,
     1.7920702601e-05,  1.7111588349e-05,  1.6300567469e-05,  1.5488149763e-05,
     1.4674839231e-05,  1.3861134879e-05,  1.3047533992e-05,  1.2234524742e-05,
     1.1422592342e-05,  1.0612218003e-05,  9.8038736906e-06,  8.9980289794e-06,
     8.1951455703e-06,  7.3956781740e-06,  6.6000753094e-06,  5.8087799945e-06,
     5.0222259747e-06,  4.2408412908e-06,  3.4650478950e-06,  2.6952553278e-06,
     1.9318705150e-06,  1.1752893268e-06,  4.2589953198e-07, -3.1591717180e-07,
    -1.0497892491e-06, -1.7753544359e-06, -2.4922577778e-06, -3.2001539672e-06,
    -3.8987079130e-06, -4.5875910177e-06, -5.2664849159e-06, -5.9350833954e-06,
    -6.5930842526e-06, -7.2402010907e-06, -7.8761532193e-06, -8.5006701688e-06,
    -9.1134919642e-06, -9.7143687730e-06, -1.0303059041e-05, -1.0879333648e-05,
    -1.1442973052e-05, -1.1993765251e-05, -1.2531511557e-05, -1.3056022025e-05,
    -1.3567114672e-05, -1.4064623334e-05, -1.4548386487e-05, -1.5018255697e-05,
    -1.5474092103e-05, -1.5915766513e-05, -1.6343159044e-05, -1.6756162746e-05,
    -1.7154678702e-05, -1.7538617258e-05, -1.7907902366e-05, -1.8262462956e-05,
    -1.8602240308e-05, -1.8927187450e-05, -1.9237262803e-05, -1.9532438832e-05,
    -1.9812695387e-05, -2.0078021512e-05, -2.0328415472e-05, -2.0563887334e-05,
    -2.0784452769e-05, -2.0990139418e-05, -2.1180984600e-05, -2.1357029905e-05,
    -2.1518330538e-05, -2.1664948346e-05, -2.1796951838e-05, -2.1914422742e-05,
    -2.2017446878e-05, -2.2106119745e-05, -2.2180543920e-05, -2.2240831923e-05,
    -2.2287099107e-05, -2.2319475175e-05, -2.2338092478e-05, -2.2343090766e-05,
    -2.2334619810e-05, -2.2312832563e-05, -2.2277889205e-05, -2.2229960317e-05,
    -2.2169217607e-05, -2.2095842963e-05, -2.2010021997e-05, -2.1911947638e-05,
    -2.1801814071e-05, -2.1679829163e-05, -2.1546197041e-05, -2.1401133411e-05,
    -2.1244857054e-05, -2.1077588914e-05, -2.0899556515e-05, -2.0710992889e-05,
    -2.0512131162e-05, -2.0303213334e-05, -2.0084482176e-05, -1.9856184304e-05,
    -1.9618568029e-05, -1.9371890077e-05, -1.9116401148e-05, -1.8852364955e-05,
    -1.8580041036e-05, -1.8299692229e-05, -1.8011585395e-05, -1.7715987582e-05,
    -1.7413166460e-05, -1.7103395216e-05, -1.6786945551e-05, -1.6464090073e-05,
    -1.6135103853e-05, -1.5800262576e-05, -1.5459837759e-05, -1.5114111039e-05,
    -1.4763354528e-05, -1.4407846390e-05, -1.4047863365e-05, -1.3683679759e-05,
    -1.3315570064e-05, -1.2943810750e-05, -1.2568674067e-05, -1.2190432878e-05,
    -1.1809360580e-05, -1.1425725372e-05, -1.1039794291e-05, -1.0651838424e-05,
    -1.0262116982e-05, -9.8708981098e-06, -9.4784407932e-06, -9.0850034306e-06,
    -8.6908412972e-06, -8.2962095884e-06, -7.9013557675e-06, -7.5065302494e-06,
    -7.1119784241e-06, -6.7179396637e-06, -6.3246534218e-06, -5.9323559025e-06,
    -5.5412730743e-06, -5.1516395889e-06, -4.7636746501e-06, -4.3776009822e-06,
    -3.9936333620e-06, -3.6119853019e-06, -3.2328603264e-06, -2.8564666919e-06,
    -2.4830013102e-06, -2.1126583213e-06, -1.7456307078e-06, -1.3821014465e-06,
    -1.0222491308e-06, -6.6625470321e-07, -3.1428344793e-07,  3.3494103224e-08,
     3.7691835819e-07,  7.1583298342e-07,  1.0500903114e-06,  1.3795403255e-06,
     1.7040472740e-06,  2.0234741402e-06,  2.3376882590e-06,  2.6465707420e-06,
     2.9499996526e-06,  3.2478620037e-06,  3.5400534208e-06,  3.8264677239e-06,
     4.1070089652e-06,  4.3815872472e-06,  4.6501169096e-06,  4.9125167274e-06,
     5.1687174724e-06,  5.4186465907e-06,  5.6622405342e-06,  5.8994474783e-06,
     6.1302089154e-06,  6.3544856311e-06,  6.5722367483e-06,  6.7834270236e-06,
     6.9880279913e-06,  7.1860190797e-06,  7.3773794892e-06,  7.5621008787e-06,
     7.7401804419e-06,  7.9116138491e-06,  8.0764100816e-06,  8.2345820910e-06,
     8.3861398492e-06,  8.5311159688e-06,  8.6695333690e-06,  8.8014294784e-06,
     8.9268432568e-06,  9.0458226500e-06,  9.1584123535e-06,  9.2646765785e-06,
     9.3646749849e-06,  9.4584747207e-06,  9.5461542772e-06,  9.6277901735e-06,
     9.7034626332e-06,  9.7732720508e-06,  9.8373036561e-06,  9.8956688137e-06,
     9.9484696902e-06,  9.9958245062e-06,  1.0037841224e-05,  1.0074657852e-05,
     1.0106389733e-05,  1.0133183552e-05,  1.0155177067e-05,  1.0172518887e-05,
     1.0185355141e-05,  1.0193857508e-05,  1.0198169785e-05,  1.0198485264e-05,
     1.0194961686e-05,  1.0187799426e-05,  1.0177165791e-05,  1.0163282130e-05,
     1.0146315524e-05,  1.0126507376e-05,  1.0104046194e-05,  1.0079171772e-05,
     1.0052096138e-05,  1.0023071665e-05,  9.9923127057e-06,  9.9600981437e-06,
     9.9266587596e-06,  9.8922779372e-06,  9.8572155084e-06,  9.8217617487e-06,
     9.7861893206e-06,  9.7508073031e-06,  9.7159210593e-06,  9.6818266732e-06,
     9.6488922037e-06,  9.6173911498e-06,  9.5877348599e-06,  9.5602016889e-06,
     9.5352460877e-06,  9.5131366561e-06,  9.4943989207e-06,  9.4792809128e-06,
     9.4683559083e-06,  9.4618896265e-06,  9.4604880509e-06,  9.4644314122e-06,
     9.4743911663e-06,  9.4906478452e-06,  9.5139045177e-06,  9.5445156124e-06,
     9.5831366280e-06,  9.6302738235e-06,  9.6864912668e-06,  9.7525195439e-06,
     9.8287035601e-06,  9.9161827518e-06,  1.0014900175e-05,  1.0126524867e-05,
     1.0250640555e-05,  1.0389249891e-05,  1.0541878290e-05,  1.0710291391e-05,
     1.0894920553e-05,  1.1095538439e-05,  1.1315643963e-05,  1.1553153971e-05,
     1.1809655138e-05,  1.2091378873e-05,  1.2397065958e-05,  1.2719058022e-05,
     1.3064550099e-05,  1.3443390954e-05,  1.4025302043e-05, -2.2207222127e-04,
     0.0000000000e+00,
};
